#include <stdlib.h>
#include <string.h>

#include "ds.h"
#include "ss.h"

void test_ds();
void test_ss(char *s, char delim);

int main(int argc, char **argv)
{
    if (argc < 2)
    {
        fprintf(stderr, "You should call this program with at least a parameter saying what to test.\n");
        fprintf(stderr, "Example: ./a3 ds 1 0\n");
        fprintf(stderr, "Example: ./a3 ss \"banana\" \"a\"\n");
        return 1;
    }

    if (strcmp(argv[1], "ds") == 0)
    {
        // Get whether to test lookup
        int include_storage;
        int include_lookup;
        if (argc == 4)
        {
            include_storage = atoi(argv[2]);
            include_lookup = atoi(argv[3]);
        }
        else
        {
            fprintf(stderr, "You need to whether to test storage and lookup (1 for yes, 0 for no).  Good-bye!\n");
            return 1;
        }

        test_ds(include_storage, include_lookup);
    }
    else if (strcmp(argv[1], "ss") == 0)
    {
        // Get the string to split and the delimiter
        char *s = NULL;
        char delim;
        if (argc == 4)
        {
            s = argv[2];
            delim = argv[3][0];
        }
        else
        {
            fprintf(stderr, "You need to provide a string and a delimiter as command-line arguments.  Good-bye!\n");
            return 1;
        }

        test_ss(s, delim);
    }

    return 0;
}

void test_ds(int test_storage, int test_lookup)
{
    // Create a few strings
    char *s1 = "hobbes";
    char *s2 = "lulu";

    // Create labels for these strings
    int s1_label = 0x78901234;
    int s2_label = 0xabcd9753;

    // Check that label-splitting works for s1's label
    int s1_id, s1_entry_num, s1_data_pos;
    split_label(s1_label, &s1_id, &s1_entry_num, &s1_data_pos);
    printf("Split s1's label (0x%x) into id=0x%x, entry=0x%x, data pos=0x%x\n",
           s1_label, s1_id, s1_entry_num, s1_data_pos);

    // If we aren't testing storage, we're done!
    if (!test_storage)
    {
        return;
    }

    // Create an empty data_store
    data_store_t data_store;
    initialize_data_store(&data_store);

    // Store the two values
    store_string(&data_store, s1_label, s1);
    store_string(&data_store, s2_label, s2);

    print_data_store(&data_store, DS_PRINT_TYPE);

    // If we aren't testing lookup, we're done!
    if (!test_lookup)
    {
        return;
    }

    // Look up substrings in the data store
    char *s3 = (char *)malloc(8 * sizeof(char));
    char *s4 = (char *)malloc(8 * sizeof(char));
    int s1_res = lookup_string(&data_store, s1_label, s3);
    int s2_res = lookup_string(&data_store, s2_label, s4);

    // Store another value, causing an ID conflict (add one to entry, clear data bits)
    int new_label = (s1_label & ~(DS_NUM_BYTES_PER_ENTRY-1)) + (DS_NUM_ENTRIES*DS_NUM_BYTES_PER_ENTRY);
    char *new_string = ":)";
    printf("\nNew label: 0x%x for data: '%s'\n", new_label, new_string);
    store_string(&data_store, new_label, new_string);

    // Look up the original string s1
    char *s5 = (char *)malloc(8 * sizeof(char));
    int s1_fail = lookup_string(&data_store, s1_label, s5);

    // Store another value that's too long
    int s6_label = 0xbad0beef;
    char *s6 = "This string is super long";
    printf("\nNew label: 0x%x for data: '%s'\n", s6_label, s6);
    store_string(&data_store, s6_label, s6);

    print_data_store(&data_store, DS_PRINT_TYPE);

    // Print out the results
    printf("\nFound strings: '%s', '%s', '%s'\n", s3, s4, s5);
    printf("Success values: %d, %d, %d\n", s1_res, s2_res, s1_fail);

    // Free the memory
    free(s3);
    free(s4);
    free(s5);
}

void test_ss(char *s, char delim)
{
    if (SS_DEBUG) printf("The string to split is: \"%s\"\n", s);
    if (SS_DEBUG) printf("The delimiter is: '%c'\n", delim);

#ifdef SS_MASTERY

    ////////////////////////
    // Mastery            //
    ////////////////////////

    // Build the array of substrings
    int num_strings;
    char **substrings = build_substrings(s, delim, &num_strings);

#else

    ////////////////////////
    // Proficiency        //
    ////////////////////////

    // Allocate space for our result, including a NULL pointer to terminate the array
    char *substrings[SS_MAX_STRINGS + 1];
    memset(substrings, 0, (SS_MAX_STRINGS+1)*sizeof(char *));

    // Split the string
    int num_strings = split_string(s, delim, substrings);

#endif // SS_MASTERY

    ////////////////////////
    // All: print results //
    ////////////////////////

    if (SS_DEBUG)
    {
        for (int i = 0; i < num_strings; i++)
        {
            printf("%d: '%s'\n", i, substrings[i]);
        }
    }

#ifdef SS_MASTERY

    ////////////////////////
    // Mastery: cleanup   //
    ////////////////////////

    // Free the array
    if (substrings)
    {
        free(substrings);
    }

#endif // SS_MASTERY
}