#ifndef __DS_H__
#define __DS_H__

#include <stdio.h>
#include <string.h>

///////////////////////////////////////////////////////////////////////////
// Useful #defines and structs -- make sure to read through these!       //
///////////////////////////////////////////////////////////////////////////

// Data store parameters: NUM_ variables must be 2^_BITS variables
#define DS_ENTRY_BITS 4
#define DS_DATA_BITS 4
#define DS_NUM_ENTRIES 16
#define DS_NUM_BYTES_PER_ENTRY 16

// Success/failure retrieving from the data store
#define DS_LOOKUP_SUCCESS 0
#define DS_LOOKUP_FAILURE -1

// Default value for all bytes in an newly initialized data store
#define DS_DEFAULT_BYTE '-'

// Constants for how to print out the data_store
#define DS_PRINT_CHARS 1
#define DS_PRINT_HEX 2
#define DS_PRINT_TYPE DS_PRINT_CHARS

// Structs for storing data -- these are important!
typedef struct
{
    int id;
    char data[DS_NUM_BYTES_PER_ENTRY];
} data_store_entry_t;

typedef struct
{
    data_store_entry_t entries[DS_NUM_ENTRIES];
} data_store_t;

///////////////////////////////////////////////////////////////////////////
// Function to define: Proficiency                                       //
///////////////////////////////////////////////////////////////////////////

/*
 * Splits an integer label into its parts: ID, entry #, and data-start position.
 * Stores these values in the appropriate provided locations.
 *
 * label: an integer whose bits contain multiple parts
 *
 * id_addr: the location to place the ID (an int) for this label
 * entry_num_addr: the location to place the entry number (an int) for this label
 * data_pos_addr: the location to place the data-start position (an int) for this label
 */
void split_label(int label, int *id_addr, int *entry_num_addr, int *data_pos_addr);

///////////////////////////////////////////////////////////////////////////
// Functions to define: Mastery                                          //
///////////////////////////////////////////////////////////////////////////

/*
 * Stores a string in the given data store.
 *
 * data_store: the data store in which to put the string
 * label: the label for the string, containing the ID, entry #, and data-start position (ignored)
 * s: the string (NULL-terminated seqeuence of chars) to store
 */
void store_string(data_store_t *data_store, int label, char *s);

/*
 * Retrives a string from the given data store.  Stores the retrieved string
 * in dest and returns LOOKUP_SUCCESS if the label matches, or has no effect
 * on dest and returns LOOKUP_FAILURE otherwise.
 *
 * data_store: the data store in which to look up the string
 * label: the label for the string, containing the ID, entry #, and data-start position
 * dest: the location in which to copy the retrieved string
 */
int lookup_string(data_store_t *data_store, int label, char *dest);

///////////////////////////////////////////////////////////////////////////
// Functions you've been given                                           //
///////////////////////////////////////////////////////////////////////////

/*
 * Initializes the given data store to have all IDs set to 0 and all data
 * bytes set to DS_DEFAULT_BYTE.
 */
void initialize_data_store(data_store_t *data_store);

/*
 * Prints the contents of the given data store, printing data either as
 * hex (print_type=PRINT_HEX) or characters (print_type=PRINT_CHARS).
 */
void print_data_store(data_store_t *data_store, int print_type);

#endif // __DS_H__