#!/usr/bin/env python

"""
This module determines if there is a space versus tab error in the user's program.

Matt Adams
Michael Domingues
Alex Voorhees

Last Modified: 17 Feb 2013
"""

class WhitespaceParser():
    """
    This is the primary class of this module. 
    It provides functionality to parse a file for whitespace inconsistancies (mixed tabs and spaces).
    """
    def __init__(self, filename):
        self.filename = filename
        self.errorLineNum = 0 # 0 if no error, otherwise line # of inconsistancy
        self.errorLineString = ""
        self.default = None
        self.trigger = None
        self.parseFile(filename)

    def reset(self):
        """This method resets the parser so it can be used on another file."""
        self.errorLineNum = 0
        self.errorLineString = ""
        self.default = None
        self.trigger = None
        
    def parseFile(self, filename):
        """This method parses the file "filename" for whitespace errors and has no returns."""
        lineNum = tabs = spaces = 0
        with open(filename) as file:
            for line in file:
                lineNum += 1
                if line.startswith(" "): 
                    spaces += 1
                    if tabs == 0:
                        self.default = "spaces"
                if line.startswith("\t"):
                    tabs += 1
                    if spaces == 0:
                        self.default = "tabs"
                if tabs > 0 and spaces > 0:
                    self.errorLineNum = lineNum
                    self.errorLineString = line
                    if line.startswith("\t"):
                        self.trigger = "tab"
                    elif line.startswith(" "):
                        self.trigger = "space"
                    break

    def hasError(self):
        """Returns True if the parser detected an error, and False otherwise."""
        return not (self.errorLineNum == 0)

    def getErrorLineNum(self):
        """Returns 0 if there is no error; otherwise returns the line number of the first whitespace problem."""
        return self.errorLineNum


    def get_error_value(self):
        if not self.hasError():
            return None
        return "File uses a mixture of tabs and spaces for indentation.\n" + \
            "          Line " + str(self.errorLineNum) + " starts with a " + self.trigger +", whereas the " + \
            "beginning of the program uses " + self.default + " for indentation."
    
    def get_error_type(self):
        if not self.hasError():
            return None
        return TabError

    def __str__(self):
        """Prints the result of the Whitespace Check."""
        if self.hasError():
            lineNum = "\033[32m%d\033[0m" % self.errorLineNum
            resultString =  'In file "' + self.filename + '", line ' + lineNum + ':' + '\n' \
                            + '  ' + self.errorLineString.strip() + '\n\n' \
                            + "\033[31mTabError:\033[0m File uses a mixture of tabs and spaces for indentation.\n" + \
                            "          Line " + str(self.errorLineNum) + " starts with a " + self.trigger +", whereas the " + \
                            "beginning of the program uses " + self.default + " for indentation."
            return resultString
        else:
            return 'File "' + self.filename + '" has consistent tabs and spaces.'


if __name__ == "__main__":
    import sys

    try:
        filename = sys.argv[1]
    except:
        print "Usage: python spaceTabCheck.py FILENAME"
        sys.exit()

    try:
        wp = WhitespaceParser(filename)
        if wp.hasError():
            print "  File \"%s\", line %d" % (filename, wp.getErrorLineNum())
            print "    " + wp.errorLineString.lstrip()
            print "TabError: Inconsistent use of tabs and spaces in indentation"
        else:
            print 'This file has consistent tabs and spaces.'
    except IOError:
        print 'Error: the file "%s" doesn\'t exist.' % filename
