'''
This module extends the functionality of the traceback module.

Rhys Lindmark
Jake Reardon
Matt Adams
Alex Voorhees
Tung Phan
Micheal Domingues

Last Modified: 24 Feb. 2013
'''
import traceback, sys, os
import code as code_filename_getter # not called, just used to see where user installed code module 
import re

def extract_tb(tb):
    '''Extract the traceback tb in the PyLearn format.'''
    lineList = []
    fullExtract = traceback.extract_tb(tb)
    fullExtract.reverse() # reversed to more easily filter out pylog stuff
    codeFilename = sys.argv[0]

    # loop through the traeback once to figure out if the traceback references multiple files
    programReferencesMultipleFiles = False
    for traceTuple in fullExtract:
        if traceTuple[0] == os.currentPyLearnPath:
            break
        else:
            filename = traceTuple[0]
            if filename != codeFilename:
                programReferencesMultipleFiles = True
    
    # loop through the traceback a second time, filtering out unwanted traceback entries, to build up our traceback
    
    for traceTuple in fullExtract:
        if traceTuple[0] == os.currentPyLearnPath:
            break
        else:
            filename, lineno, moduleName, code = traceTuple
            if not (moduleName == '<module>' and code == None):
                if filename == '<console>':
                    line = get_console_tb_entry_str(lineno, moduleName=moduleName)
                else:
                    if programReferencesMultipleFiles:
                        line = get_tb_entry_str(os.path.relpath(filename), lineno, code, moduleName=moduleName)
                    else:
                        line = get_one_file_tb_entry_str(lineno, code, moduleName=moduleName)
                lineList.append(line)
    if sys.isConsole:
        lineList.pop() # in console mode, the last tb entry is always in the code module (as per the code module docs)
    lineList.reverse() # re-reverse the line list, so that the most recent calls are last

    return lineList


def get_one_file_tb_entry_str(lineno, code, moduleName='<module>'):
    '''Returns a formatted string of one entry in a traceback.'''
    if moduleName == '<module>':
        return 'line \033[1;32m%d\033[0m:\n  %s\n' % (lineno, code)
    else:
        return '(in \'%s\'), line \033[1;32m%d\033[0m:\n  %s\n' % (moduleName, lineno, code)

def get_tb_entry_str(filename, lineno, code, moduleName='<module>'):
    '''Returns a formatted string of one entry in a traceback.'''
    if moduleName == '<module>':
        return 'File "%s", line \033[1;32m%d\033[0m:\n  %s\n' % (filename, lineno, code)
    else:
        return 'File "%s" (in \'%s\'), line \033[1;32m%d\033[0m:\n  %s\n' % (filename, moduleName, lineno, code)
        
def get_console_tb_entry_str(lineno, code=None, moduleName='<module>'):
    '''Returns a formatted string of one entry in a traceback for entries coming from <console>.'''
    entry_str = ''
    if moduleName == '<module>':
        entry_str += 'line \033[1;32m%d\033[0m' % (lineno)
    else:
        entry_str += 'called \'%s\'' % (moduleName)
    if code != None:
        entry_str += ':\n  ' + code + '\n'
    return entry_str 
    
def get_syntax_str():
    '''
    Specialized method for printing the syntax error tracback entry. 
    This is a work-around for the fact that because python doesn't include
    the last syntax tb entry in the traceback itself.
    '''
    # This does not need to be printed in the console.
    if sys.isConsole:
        return ''
    lines = traceback.format_exception_only(sys.last_type, sys.last_value)
    filename = lines[0][8:lines[0].find('"',8)]
    lineNumber = int(lines[0].split()[-1])
    code = lines[1].strip()
    offset = len(lines[1]) - len(lines[1].lstrip())
    return get_tb_entry_str(filename, lineNumber, code)+lines[2][offset-2:-1]

def extract_tb_str(tb):
    '''Return a string of the extract_tb results.'''
    tbString = '\n'.join((str(line) for line in extract_tb(tb)))
    
    return tbString


def formatErrMsg(eType, eValue):
    '''This function formats/colors the final line of an error.'''
    if type(eType) != str:
        try:
            eType = eType.__name__
        except:
            eType = 'Error'
    if sys.isConsole:
        if str(eValue).strip() == '':
            return '\033[1;31m{}\033[0m\n'.format(eType)
        else:
            return '\033[1;31m%s:\033[0m %s\n' % (eType, eValue)
    else:
        if str(eValue).strip() == '':
            return '\n\033[1;31m{}\033[0m'.format(eType)
        else:
            return '\n\033[1;31m%s:\033[0m %s' % (eType, eValue)

 
class ExecStack:
    def __init__(self, trace):
        self.stackList = self._build_stack(trace)
        
    def _build_stack(self, trace):
        '''Return the stack "seen" by the user's program. (PyLearn stack frames are filtered out.)'''
        # useful: http://code.activestate.com/recipes/52215-get-more-information-from-tracebacks/
        tb = trace
        # loop to get the final traceback
        while tb.tb_next:
            tb = tb.tb_next
        # loop to fill the stack with non-pylearn frames
        stack = []
        curFrame = tb.tb_frame
        while curFrame.f_code.co_filename !=  os.currentPyLearnPath:
            stack.append(curFrame)
            curFrame = curFrame.f_back
        return stack

    def get_err_lineno(self):
        '''Return the line number the error occured on.'''
        return self.stackList[0].f_lineno

    def get_err_line(self):
        '''Return the line the error occured on.'''
        if sys.isConsole:
            return ''
        else:
             err_lineno = self.get_err_lineno()-1

             with open(self.stackList[0].f_code.co_filename, 'r') as progFile:
                 i = 0
                 for line in progFile:
                     if i == err_lineno:
                         break
                     i += 1
             return line.strip()

    def get_var_names_in_scope(self):
        '''Return a list of all of the defined variable names.'''
        var_names = []
        for frame in self.stackList:
            calledFile = sys.argv[0]
            if frame.f_code.co_filename == calledFile or frame.f_code.co_filename == '<console>':
                for key, value in frame.f_locals.items():
                    var_names.append(key)
        return var_names
        
    def get_value_of_variable(self, userVar):
        for frame in self.stackList:
            for key, value in frame.f_locals.items():
                if key == userVar:
                    return value
        
                
    def __str__(self):
        return '\n'.join(str((f.f_code.co_name, f.f_code.co_filename, f.f_lineno)) for f in self.stackList)
