''' 
Modified version of Pypy parser to handle some syntax error

Tung Phan
(small changes by Rhys and Matt)

Last Modified: 17 Feb. 2013
'''

from pyparser import parser, pytokenizer, pygram, error
import consts
import sys
from errorTree import tbTools

class CompileInfo(object):
    """Stores information about the source being compiled.

    * filename: The filename of the source.
    * mode: The parse mode to use. ('exec', 'eval', or 'single')
    * flags: Parser and compiler flags.
    * encoding: The source encoding.
    * last_future_import: The line number and offset of the last __future__
      import.
    * hidden_applevel: Will this code unit and sub units be hidden at the
      applevel?
    """

    def __init__(self, filename, mode="exec", flags=0, future_pos=(0, 0),
                 hidden_applevel=False):
        self.filename = filename
        self.mode = mode
        self.encoding = None
        self.flags = flags
        self.last_future_import = future_pos
        self.hidden_applevel = hidden_applevel


_targets = {
'eval' : pygram.syms.eval_input,
'single' : pygram.syms.single_input,
'exec' : pygram.syms.file_input,
}

class PythonParser(parser.Parser):
    '''
    Modified PythonParser from Pypy to handle more type of Syntax Errors
    '''
    def __init__(self, grammar = pygram.python_grammar):
        parser.Parser.__init__(self, grammar)
    

    def parse_source(self, textsrc, compile_info):
        '''
        Parse the source code to check for Syntax Error
        If pass, return None. Otherwise return the error
        '''
        source_lines = textsrc.splitlines(True)

        flags = compile_info.flags

        source_lines = textsrc.splitlines(True)
        if source_lines and not source_lines[-1].endswith("\n"):
            source_lines[-1] += '\n'

        if textsrc and textsrc[-1] == "\n":
            flags &= ~consts.PyCF_DONT_IMPLY_DEDENT

        self.prepare(_targets[compile_info.mode])
        tp = 0
        try_list = []
        last_try = None
        
        lineDict = {}
        
        try:
            try:
                tokens = pytokenizer.generate_tokens(source_lines, flags)
                
                for tp, value, lineno, column, line in tokens:
                    if lineno in lineDict:
                        lineDict[lineno].append(value)
                    else:
                        lineDict[lineno] = [value]
                
                    if self.add_token(tp, value, lineno, column, line):
                        break
                    #handle try/except
                    if (value.lower()=="try"):
                        try_list.append((value, lineno, column))
                    elif (value.lower()=="except"):
                        if last_try:
                            if (last_try[2] != column):
                                last_try = try_list.pop()
                        else:
                            last_try = try_list.pop()
                    
                    elif (value.lower()=="finally"):
                        while (try_list[-1][2] != column):
                            try_list.pop()
                        try_list.pop()
                        
            except error.TokenError, e:
                #handle unclosed/mixmatched parenthesis
                if "parenthesis" in e.msg:
                    matched_paren = {'(':')', '[':']', '{':'}'}
                    open_sym = e.msg.split()[1]
                    e.msg = "'%s' must be closed by a matching '%s'" % (open_sym, matched_paren[open_sym])
                    e.filename = compile_info.filename
                    return e
                elif "mixmatched" in e.msg:
                    matched_paren = {')':'(', ']':'[', '}':'{'}
                    closed_sym = e.msg.split()[1]
                    e.msg = "'%s' must close a matching '%s'" % (closed_sym, matched_paren[closed_sym])
                    return e
                elif (e.msg != "invalid syntax"):
                    new_err = SyntaxError(e.msg)
                    new_err.lineno = e.lineno
                    new_err.filename = e.filename
                    new_err.text = e.text
                    return new_err
            except error.TokenIndentationError, e:
                new_err = IndentationError(e.msg)
                new_err.lineno = e.lineno
                new_err.filename = e.filename
                new_err.text = e.text
                return new_err
            except parser.ParseError, e:
                import keyword
                tokens_on_err_line = lineDict[e.lineno]
                
                #Is there a reason why if, not elif?
                #Invalid syntax. (vs. no caps, no period)
                msg = None
                for possible_keyword in tokens_on_err_line:
                    for real_keyword in keyword.kwlist:
                        if real_keyword == possible_keyword.lower() and real_keyword != possible_keyword:
                            msg = "Special,{},{}".format(real_keyword, possible_keyword)
                
                
                # Catch parse errors, pretty them up and reraise them as a
                # SyntaxError.
                new_err = error.SyntaxError
                if (e.value.lower() in ('except', 'finally')):
                    if len(try_list)>0:
                        msg = "'%s' in line %d does not match indentation level of current 'try' at line %d" % (e.value.lower(), e.lineno, try_list[-1][1])
                    else:
                        msg = "Unexpected %s. except/finally must go with try block" % (e.value.lower())
                if (len(e.states) == 2):
                    expected_ids = [st[0] for st in e.states]
                    #159 = id of except_clause and 160 = id of finally
                    if (159 in expected_ids and 160 in expected_ids):
                        msg = "Unclosed 'try' block opened on line %d." % (try_list[-1][1])
                if (e.token_type == 22 and e.expected in [8, 11, 10, 27]):
                    '''
                    Handle syntax error with =
                    ids_8 = )
                    ids_11 = :
                    ids_10 = ]
                    ids_27 = }
                    '''
                    msg = "Equals sign issue. a = b assigns b to a. a == b checks to see if a is equal to b."
                if msg:
                    return new_err(msg, e.lineno, e.column, e.line,
                              compile_info.filename)

            else:
                tree = self.root
        finally:
            # Avoid hanging onto the tree.
            self.root = None

        return None


def SyntaxChecker(filename=None, code_string=None):
    if filename != None and code_string == None:
        info = CompileInfo(filename)
        text = open(filename).read()
        ps = PythonParser()
        res = ps.parse_source(text, info)       
    elif filename == None and code_string != None:
        info = CompileInfo('<console>')
        ps = PythonParser()
        res = ps.parse_source(code_string, info)
    return res

def printFormattedSyntaxMessage(syntaxError):
    if syntaxError.filename == '<console>':
        msg = tbTools.get_console_tb_entry_str(syntaxError.lineno, syntaxError.text.strip())
    else:
        msg = tbTools.get_tb_entry_str(syntaxError.filename, syntaxError.lineno, syntaxError.text.strip())
    offset = syntaxError.offset - (len(syntaxError.text) - len(syntaxError.text.lstrip()))
    if "Special" in syntaxError.msg: 
        nothing, real_keyword, possible_keyword = syntaxError.msg.split(",")
        msg += tbTools.formatErrMsg("Syntax Error", "Invalid syntax.")
        msg += tbTools.formatErrMsg("Possible Keyword Typo", "Perhaps you meant to type '{}' instead of '{}'.".format(real_keyword, possible_keyword))
    else:
        msg += " " * (offset+2) + "^" + "\n"
        msg += tbTools.formatErrMsg("Syntax Error", syntaxError.msg)
    return msg
