'''
Value Error and subnodes

Rhys Lindmark
Matthew Adams
Last Modified: January 5, 2012

'''

from .. import tbTools 
from ..BaseErrorNode import ErrorNode
import re

'''
Base ValueError node

'''

class ValueErrorNode(ErrorNode):
    def gdn(self):
        if re.match("invalid literal for .+: '.*'", str(self.errValue)):
            return InvalidLiteralNode(self.errType, self.errValue, self.tb)
        elif str(self.errValue) == 'list.remove(x): x not in list':
            return ListDotRemoveNode(self.errType, self.errValue, self.tb)
        elif "range() step argument must not be zero" == str(self.errValue):
            return StepArgumentMustNotBeZeroNode(self.errType, self.errValue, self.tb)
        elif "I/O operation on closed file" in str(self.errValue):
            return OperationOnClosedFileNode(self.errType, self.errValue, self.tb)
        elif re.match("need more than \d values to unpack$", str(self.errValue)):
            return NeedMoreThanXValuesToUnpackNode(self.errType, self.errValue, self.tb)
        elif "too many values to unpack" == str(self.errValue):
            return TooManyValuesToUnpackNode(self.errType, self.errValue, self.tb)
        else:
            return None

    def __str__(self):
        return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('ValueError', self.errValue)


'''
Nodes of the form 'invalid literal for ***:'

'''

class BaseInvalidLiteralNode(ErrorNode):
    def __init__(self, errType, errValue, tb):
        ErrorNode.__init__(self, errType, errValue, tb)
        err_val_parser = re.match("invalid literal for (?P<func>.+): (?P<lit>'.*')", str(self.errValue))
        self.invalid_literal = err_val_parser.group('lit')
        self.attempted_call = err_val_parser.group('func')


class InvalidLiteralNode(BaseInvalidLiteralNode):
    def gdn(self):
        if self.attempted_call[0:16] == 'int() with base ':
            return IntInvalidLiteralNode(self.errType, self.errValue, self.tb)
        else:
            return None

    def __str__(self):
        return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('ValueError', "{} cannot be called on {}.".format(self.attempted_call, self.invalid_literal))


class IntInvalidLiteralNode(BaseInvalidLiteralNode):
    def __str__(self):
        base = self.attempted_call[16:].strip()
        if base == '10':
            base_part_of_msg = 'an integer'
        else:
            base_part_of_msg = 'an integer. {} is not a string of a base {} integer'.format(self.invalid_literal, base)
        return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('ValueError', "int() cannot convert {} into {}.".format(self.invalid_literal, base_part_of_msg))
        
    def gdn(self):
        literal_is_float_str = True
        try:
            float(self.invalid_literal.strip("'"))
        except:
            literal_is_float_str = False
        if literal_is_float_str:
            return TriedToIntAFloatStringNode(self.errType, self.errValue, self.tb)
        else:
            return None
        #Other possible one: make sure you sanitize raw_input


class TriedToIntAFloatStringNode(BaseInvalidLiteralNode):
    def __str__(self):
        #Telling them what to do is sketch in general, but not terrible here.
        return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('ValueError', "int() cannot convert {} into an integer. First turn it into a float.".format(self.invalid_literal))




'''
Nodes not of the form 'invalid literal for ***:'

'''

class ListDotRemoveNode(ErrorNode):
    def __str__(self):
        try:
            itemThatUserIsTryingToRemove = re.search('.remove\(\w+\)', self.execStack.get_err_line()).group(0)[8:-1]
            return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('ValueError', "{item} is not in the list, so {item} cannot be removed.".format(item=itemThatUserIsTryingToRemove))
        except:
            return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('ValueError', "Attempted to remove an item from a list that it wasn't in.")

        
class StepArgumentMustNotBeZeroNode(ErrorNode):
    def __str__(self):
        # Potentially useful for extension: zeroStep = re.search('range\(\s*.+\s*,\s*.+\s*,\s*(?P<step>.+)\s*\)', self.execStack.get_err_line()).group('step')
        return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('ValueError', "The third argument to range must be a non-zero integer.")

        
class OperationOnClosedFileNode(ErrorNode):
    def __str__(self):
        return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('ValueError', "Attempted to read from or write to a closed file.")


class NeedMoreThanXValuesToUnpackNode(ErrorNode):
    def __str__(self):
        try:
            variablesToBeAssigned = re.findall('[^=]*=', self.execStack.get_err_line())[0]
            commaCount = 0
            for character in variablesToBeAssigned:
                if character == ",":
                    commaCount += 1
            numberOfVariablesGiven = str(self.errValue)[15]

            return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('ValueError', "Trying to assign values to {} variables, but only have {} values.".format(commaCount + 1, numberOfVariablesGiven))
        except:
            return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('ValueError', "Trying to assign values variables, but there are fewer values than variables.")

class TooManyValuesToUnpackNode(ErrorNode):
    def __str__(self):
        try:
            variablesToBeAssigned = re.findall('[^=]*=', self.execStack.get_err_line())[0]
            commaCount = 0
            for character in variablesToBeAssigned:
                if character == ",":
                    commaCount += 1
            return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('ValueError', "Trying to assign more than {cc} values to only {cc} variables.".format(cc=commaCount + 1))
        except:
            return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('ValueError', "Trying to assign more values than you have variables.")
