"""
Type Error and subnodes.

Rhys Lindmark
Matt Adams
Alex Voorhees

Last modified 1/29/12
"""

from .. import tbTools
from ..BaseErrorNode import ErrorNode
import re

class TypeErrorNode(ErrorNode):
    """
    Note; the following errors are bomb and should be kept:
    'list indicies must be integer, not other type of object.'
    'function takes 2 arguments, (one given)'
    """
    def __str__(self):
        return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('TypeError', self.errValue)
    def gdn(self):
        if "object is not iterable" in str(self.errValue):
            return NotIterableNode(self.errType, self.errValue, self.tb)
        elif "unsupported operand" in str(self.errValue):
            return UnsupportedOperandNode(self.errType, self.errValue, self.tb)
        elif "cannot concatenate" in str(self.errValue):
            return CannotConcatenateStringWithOtherNode(self.errType, self.errValue, self.tb)
        elif "can only concatenate" in str(self.errValue):
            return CanOnlyConcatenateToListOrTupleNode(self.errType, self.errValue, self.tb)
        elif ("subscriptable" in str(self.errValue)) or ("__getitem__" in str(self.errValue)):
            return GetItemNode(self.errType, self.errValue, self.tb)
        elif "no len()" in str(self.errValue):
            return CantUseLengthNode(self.errType, self.errValue, self.tb)
        elif "expected a character buffer object" == str(self.errValue):
            return CharBufferNode(self.errType, self.errValue, self.tb)
        elif "__init__() should return None" in str(self.errValue):
            return InitShouldReturnNoneNode(self.errType, self.errValue, self.tb)
        elif "object is not callable" in str(self.errValue):
            return ObjectIsNotCallableNode(self.errType, self.errValue, self.tb)

class NotIterableNode(ErrorNode):
    def __str__(self):
        objectType = re.search(r"'\w+'", str(self.errValue)).group(0)
        return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('TypeError', "Can't loop over an object of type -- " + objectType + ".")
    def gdn(self):
        objectType = re.search(r"'\w+'", str(self.errValue)).group(0)
        myExecStack = tbTools.ExecStack(self.tb)
        loopingVariable = re.search(r"for \w+", myExecStack.get_err_line()).group(0)[4:]
        if objectType == "'int'" and len(loopingVariable) == 1:
            return NotIterableWantedRangeNode(self.errType, self.errValue, self.tb)

class NotIterableWantedRangeNode(ErrorNode):
    def __str__(self):
        return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('TypeError', "Can't loop over an object of type -- 'int'.") + tbTools.formatErrMsg("Possible Range Issue", "Use range(10) to loop 10 times.")

class UnsupportedOperandNode(ErrorNode):
    def __str__(self):
        object1Type = re.findall(r"'\w+'", str(self.errValue))[0]
        object2Type = re.findall(r"'\w+'", str(self.errValue))[1]
        badOperand = re.search(r"\S+:", str(self.errValue)).group(0)[:-1]
        # Suggestion: "Can't execute: 'int' type + 'str' type." as discussed in email thread
        return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('TypeError', "Can't execute: " + object1Type + " type " + badOperand + " " + object2Type + " type.")

class CannotConcatenateStringWithOtherNode(ErrorNode):
    def __str__(self):
        object1Type = re.findall(r"'\w+'", str(self.errValue))[0]
        object2Type = re.findall(r"'\w+'", str(self.errValue))[1]
        badOperand = '+'
        # Suggestion: "Can't execute: 'int' type + 'str' type." as discussed in email thread
        return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('TypeError', "Can't execute: " + object1Type + " type " + badOperand + " " + object2Type + " type.")

class CanOnlyConcatenateToListOrTupleNode(ErrorNode):
     def __str__(self):
        object1Type = re.search(r"\w+ \(", str(self.errValue)).group(0)[:-2]
        object2Type = re.search(r'"\w+"', str(self.errValue)).group(0)
        badOperand = '+'
        # Suggestion: "Can't execute: 'int' type + 'str' type." as discussed in email thread
        return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('TypeError', "Can't execute: " + object1Type + " type " + badOperand + " " + object2Type + " type.")

class GetItemNode(ErrorNode):
    def __str__(self):
        objectType = re.search(r"'\w+'", str(self.errValue)).group(0)
        return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('TypeError', "Can't access an index for an object of type -- " + objectType + ".") 
    def gdn(self):
        # Note: I added the try/except because I can think of a couple (unlikely but possible) ways this could break.
        if "'function'" in str(self.errValue) or "'builtin_function_or_method'" in str(self.errValue):
            try:
                return SquareBracketsForFunctionCallNode(self.errType, self.errValue, self.tb)
            except:
                return None

class SquareBracketsForFunctionCallNode(ErrorNode):
    def __str__(self):
        return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('TypeError', "Call a function with () instead of [].")
        
class CantUseLengthNode(ErrorNode):
    def __str__(self):
        objectType = re.search(r"'\w+'", str(self.errValue)).group(0)
        return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('TypeError', "Object of type " + objectType + " has no len().") 

class CharBufferNode(ErrorNode):
    def __str__(self):
        # Potential sub-node: return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('TypeError', "Only strings can be written to files.")
        return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('TypeError', "Function expected a string as an argument.")  
        

class InitShouldReturnNoneNode(ErrorNode):
    def __str__(self):
        return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('TypeError', "__init__() should return None.") 

class ObjectIsNotCallableNode(ErrorNode):
    def __str__(self):
        objectType = re.search(r"'\w+'", str(self.errValue)).group(0)
        # Suggestion: objects of type 'int' cannot be called like functions
        #             I think it's a bit clearer and more accurate.
        return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('TypeError', "Objects of type " + objectType + " cannot be called like a function.") 
    def gdn(self):
        myExecStack = tbTools.ExecStack(self.tb)
        instanceVariableCalled = len(re.findall(r"\w+.\w+\(\)", myExecStack.get_err_line())) == 1
        if instanceVariableCalled:
            return InstanceVariableCalledNode(self.errType, self.errValue, self.tb)

class InstanceVariableCalledNode(ErrorNode):
    def __str__(self):
        myExecStack = tbTools.ExecStack(self.tb)
        instanceVariable = re.findall(r"\w+.\w+\(\)", myExecStack.get_err_line())[0][:-2]
        objectType = re.search(r"'\w+'", str(self.errValue)).group(0)
        # Suggestion: This one is misleading- the problem isn't that the var is an instance variable. 
        #             Instance variables can totally contain functions and therefore be callable (we just call them methods). 
        #             The real problem is that the instance var references something other than a function.
        
        #Matt's suggestion above was debated for many (40) days and nights. In the end, we decided to keep the message as is -- techically wrong (because methods are actually instance variables), but more helpful to a beginner programmer.
        return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('TypeError', "Objects of type " + objectType + " cannot be called like a function.".format(objectType)) + tbTools.formatErrMsg("Possible Instance Variable Issue", "'{}' is an instance variable and cannot be called like a function.".format(instanceVariable))





