'''
NameError and subnodes.

Matt Adams
Michael Domingues
Rhys Lindmark

Last Modified: 2/23/2013

'''

from .. import tbTools
from ..BaseErrorNode import ErrorNode
import re

class BaseNameErrorNode(ErrorNode):
    def __init__(self, errType, errValue, tb, similarVar=None, verb=False):
        ErrorNode.__init__(self, errType, errValue, tb)
        try:
            self.badVar = re.search(r"'\w+'",str(self.errValue)).group(0).strip("'")
        except:
            self.badVar = ''
        self.similarVar = similarVar
        self.verb = verb
        self.mainStr = tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('NameError',"The variable '{}' wasn\'t defined when you tried to access it.".format(self.badVar))

class NameErrorNode(BaseNameErrorNode):
    def getMinEditDist(self, seq1, seq2):
        '''Computes the min edit distance between two sequences. Source: http://en.wikibooks.org/wiki/Algorithm_Implementation/Strings/Levenshtein_distance#Python'''
        oneago = None
        thisrow = range(1, len(seq2) + 1) + [0]
        for x in xrange(len(seq1)):
            twoago, oneago, thisrow = oneago, thisrow, [0] * len(seq2) + [x + 1]
            for y in xrange(len(seq2)):
                delcost = oneago[y] + 1
                addcost = thisrow[y - 1] + 1
                subcost = oneago[y - 1] + (seq1[x] != seq2[y])
                thisrow[y] = min(delcost, addcost, subcost)
        return thisrow[len(seq2) - 1]
    
    def gdn(self):
        if self.badVar == '':
            return None
        definedVariables = self.execStack.get_var_names_in_scope()
        
        for definedVariable in definedVariables:
            if hasattr(self.execStack.get_value_of_variable(definedVariable), '__call__'):
                myVerb = 'call'
            else:
                myVerb = 'access'
            minEditDistance = self.getMinEditDist(definedVariable, self.badVar)
            if definedVariable.lower() == self.badVar.lower() and definedVariable != self.badVar:
                return NameErrorCapNode(self.errType, self.errValue, self.tb, similarVar=definedVariable, verb=myVerb)
            elif ((definedVariable + 's') == self.badVar) or (definedVariable == (self.badVar + 's')):
                return NameErrorPluralNode(self.errType, self.errValue, self.tb, similarVar=definedVariable, verb=myVerb)
            elif ((3 <= len(self.badVar) <= 4) and minEditDistance == 1) or ((5 <= len(self.badVar) <= 6) and 1 <= minEditDistance <= 2) or ((len(self.badVar) > 6) and 1 <= minEditDistance <= 3):
                return NameErrorMinEditNode(self.errType, self.errValue, self.tb, similarVar=definedVariable, verb=myVerb)
        return None

    def __str__(self):
        if re.match(".*name '\w+' is not defined", str(self.errValue)):
            return self.mainStr
        else:
            return tbTools.extract_tb_str(self.tb) + tbTools.formatErrMsg('NameError', self.errValue)

class NameErrorCapNode(BaseNameErrorNode):
    def __str__(self):
        return self.mainStr + tbTools.formatErrMsg('Possible Capitalization Issue', "Perhaps you meant to {} '{}' instead of '{}'.".format(self.verb, self.similarVar, self.badVar))

class NameErrorPluralNode(BaseNameErrorNode):
    def __str__(self):
        return self.mainStr + tbTools.formatErrMsg('Possible Plural Issue', "Perhaps you meant to {} '{}' instead of '{}'.".format(self.verb, self.similarVar, self.badVar))

class NameErrorMinEditNode(BaseNameErrorNode):
    def __str__(self):
        return self.mainStr + tbTools.formatErrMsg('Possible Typo', "Perhaps you meant to {} '{}' instead of '{}'.".format(self.verb, self.similarVar, self.badVar))
