//
//  ASIHTTPRequestTests.m
//  Part of ASIHTTPRequest -> http://allseeing-i.com/ASIHTTPRequest
//
//  Created by Ben Copsey on 01/08/2008.
//  Copyright 2008 All-Seeing Interactive. All rights reserved.
//

#import "ASIHTTPRequestTests.h"
#import "ASIHTTPRequest.h"
#import "ASINSStringAdditions.h"
#import "ASINetworkQueue.h"
#import "ASIFormDataRequest.h"
#import <SystemConfiguration/SystemConfiguration.h>

// Used for subclass test
@interface ASIHTTPRequestSubclass : ASIHTTPRequest {}
@end
@implementation ASIHTTPRequestSubclass;

// For testing exceptions are caught
- (void)startRequest
{
	[[NSException exceptionWithName:@"Test Exception" reason:@"Test Reason" userInfo:nil] raise];
}
@end

@implementation ASIHTTPRequestTests

- (void)testBasicDownload
{
	NSURL *url = [NSURL URLWithString:@"http://allseeing-i.com"];
	ASIHTTPRequest *request = [ASIHTTPRequest requestWithURL:url];
	[request startSynchronous];
	NSString *html = [request responseString];
	GHAssertNotNil(html,@"Basic synchronous request failed");

	// Check we're getting the correct response headers
	NSString *pingBackHeader = [[request responseHeaders] objectForKey:@"X-Pingback"];
	BOOL success = [pingBackHeader isEqualToString:@"http://allseeing-i.com/Ping-Back"];
	GHAssertTrue(success,@"Failed to populate response headers");
	
	// Check we're getting back the correct status code
	url = [[[NSURL alloc] initWithString:@"http://allseeing-i.com/a-page-that-does-not-exist"] autorelease];
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request startSynchronous];
	success = ([request responseStatusCode] == 404);
	GHAssertTrue(success,@"Didn't get correct status code");	
	
	// Check data is as expected
	NSRange notFound = NSMakeRange(NSNotFound, 0);
	success = !NSEqualRanges([html rangeOfString:@"All-Seeing Interactive"],notFound);
	GHAssertTrue(success,@"Failed to download the correct data");
	
	// Attempt to grab from bad url
	url = [[[NSURL alloc] initWithString:@""] autorelease];
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request startSynchronous];
	success = [[request error] code] == ASIInternalErrorWhileBuildingRequestType;
	GHAssertTrue(success,@"Failed to generate an error for a bad host");
	
	request = [[[ASIHTTPRequest alloc] initWithURL:nil] autorelease];
	[request startSynchronous];
	success = [[request error] code] == ASIUnableToCreateRequestErrorType;
	GHAssertTrue(success,@"Failed to generate an error for a bad host");
}

- (void)testCancel
{
	// We run this test on the main thread because otherwise we can't depend on the  delegate being notified before we need to test it's working
	[self performSelectorOnMainThread:@selector(runCancelTest) withObject:nil waitUntilDone:YES];

}

- (void)runCancelTest
{
	ASIHTTPRequest *request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/the_great_american_novel_%28abridged%29.txt"]];
	[request startAsynchronous];
	[request cancel];
	GHAssertNotNil([request error],@"Failed to cancel the request");
	
	// Test cancelling a redirected request works
	// This test is probably unreliable on very slow or very fast connections, as it depends on being able to complete the first request (but not the second) in under 2 seconds
	request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/cancel_redirect"]];
	[request startAsynchronous];
	
	[[NSRunLoop currentRunLoop] runUntilDate:[NSDate dateWithTimeIntervalSinceNow:2.0]];
	[request cancel];
	
	BOOL success = ([[[request url] absoluteString] isEqualToString:@"http://allseeing-i.com/ASIHTTPRequest/tests/the_great_american_novel.txt"]);

	GHAssertTrue(success, @"Request did not redirect quickly enough, cannot proceed with test");
	
	GHAssertNotNil([request error],@"Failed to cancel the request");	 
	
	success = [request totalBytesRead] < 7900198;
	GHAssertTrue(success, @"Downloaded the whole of the response even though we should have cancelled by now");
	

}



- (void)testDelegateMethods
{
	// We run this test on the main thread because otherwise we can't depend on the  delegate being notified before we need to test it's working
	[self performSelectorOnMainThread:@selector(performDelegateMethodsTest) withObject:nil waitUntilDone:YES];
}

- (void)performDelegateMethodsTest
{
	started = NO;
	finished = NO;
	failed = NO;
	
	// Test default delegate methods
	ASIHTTPRequest *request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com"]];
	[request setDelegate:self];
	[request startSynchronous];
	
	// Hacky, but this test won't run on the main thread, we have to hope the delegate methods will be called in this time
	[[NSRunLoop currentRunLoop] runUntilDate:[NSDate dateWithTimeIntervalSinceNow:2]];
	
	GHAssertTrue(started,@"Failed to call the delegate method when the request started");	
	GHAssertTrue(finished,@"Failed to call the delegate method when the request finished");
	
	request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/the_great_american_novel_%28abridged%29.txt"]];
	[request setDelegate:self];
	[request setTimeOutSeconds:0.01];
	[request startSynchronous];
	
	GHAssertTrue(failed,@"Failed to call the delegate method when the request failed");
	
	started = NO;
	finished = NO;
	failed = NO;
	
	// Test custom delegate methods
	request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com"]];
	[request setDelegate:self];
	[request setDidStartSelector:@selector(delegateTestStarted:)];
	[request setDidFinishSelector:@selector(delegateTestFinished:)];
	[request startSynchronous];
	

	GHAssertTrue(started,@"Failed to call the delegate method when the request started");	
	GHAssertTrue(finished,@"Failed to call the delegate method when the request finished");
	
	request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/the_great_american_novel_%28abridged%29.txt"]];
	[request setDidFailSelector:@selector(delegateTestFailed:)];
	[request setDelegate:self];
	[request setTimeOutSeconds:0.01];
	[request startSynchronous];
	
	GHAssertTrue(failed,@"Failed to call the delegate method when the request failed");
	
}

- (void)requestStarted:(ASIHTTPRequest *)request
{
	started = YES;
}

- (void)requestFinished:(ASIHTTPRequest *)request
{
	finished = YES;
}

- (void)requestFailed:(ASIHTTPRequest *)request
{
	failed = YES;
}

- (void)delegateTestStarted:(ASIHTTPRequest *)request
{
	started = YES;
}

- (void)delegateTestFinished:(ASIHTTPRequest *)request
{
	finished = YES;
}

- (void)delegateTestFailed:(ASIHTTPRequest *)request
{
	failed = YES;
}

- (void)testConditionalGET
{
	ASIHTTPRequest *request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com/i/logo.png"]];
	[request startSynchronous];
	BOOL success = ([request responseStatusCode] == 200);
	GHAssertTrue(success, @"Failed to download file, cannot proceed with this test");
	success = ([[request responseData] length] > 0);
	GHAssertTrue(success, @"Response length is 0, this shouldn't happen");
	
	NSString *etag = [[request responseHeaders] objectForKey:@"Etag"];
	NSString *lastModified = [[request responseHeaders] objectForKey:@"Last-Modified"];
	
	GHAssertNotNil(etag, @"Response didn't return an etag");
	GHAssertNotNil(lastModified, @"Response didn't return a last modified date");
	
	request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com/i/logo.png"]];
	[request addRequestHeader:@"If-Modified-Since" value:lastModified];
	[request addRequestHeader:@"If-None-Match" value:etag];
	[request startSynchronous];
	success = ([request responseStatusCode] == 304);
	GHAssertTrue(success, @"Got wrong status code");
	success = ([[request responseData] length] == 0);
	GHAssertTrue(success, @"Response length is not 0, this shouldn't happen");
	
}

- (void)testException
{
	ASIHTTPRequestSubclass *request = [ASIHTTPRequestSubclass requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com"]];
	[request startSynchronous];
	NSError *error = [request error];
	GHAssertNotNil(error,@"Failed to generate an error for an exception");
	
	BOOL success = [[[error userInfo] objectForKey:NSLocalizedDescriptionKey] isEqualToString:@"Test Exception"];
	GHAssertTrue(success, @"Generated wrong error for exception");
	
}

- (void)testCharacterEncoding
{
	
	NSArray *IANAEncodings = [NSArray arrayWithObjects:@"UTF-8",@"US-ASCII",@"ISO-8859-1",@"UTF-16",nil];
	NSUInteger NSStringEncodings[] = {NSUTF8StringEncoding,NSASCIIStringEncoding,NSISOLatin1StringEncoding,NSUnicodeStringEncoding};
	
	NSUInteger i;
	for (i=0; i<[IANAEncodings count]; i++) {
		NSURL *url = [[[NSURL alloc] initWithString:[NSString stringWithFormat:@"http://allseeing-i.com/ASIHTTPRequest/tests/Character-Encoding/%@",[IANAEncodings objectAtIndex:i]]] autorelease];
		ASIHTTPRequest *request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
		[request startSynchronous];
		BOOL success = [request responseEncoding] == NSStringEncodings[i];
		GHAssertTrue(success,[NSString stringWithFormat:@"Failed to use the correct text encoding for %@i",[IANAEncodings objectAtIndex:i]]);
	}
					 
	NSURL *url = [[[NSURL alloc] initWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/Character-Encoding/Something-else"] autorelease];
	ASIHTTPRequest *request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setDefaultResponseEncoding:NSWindowsCP1251StringEncoding];
	[request startSynchronous];
	BOOL success = [request responseEncoding] == [request defaultResponseEncoding];
	GHAssertTrue(success,[NSString stringWithFormat:@"Failed to use the default string encoding"]);
	
	// Will return a Content-Type header with charset in the middle of the value (Fix contributed by Roman Busyghin)
	url = [[[NSURL alloc] initWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/Character-Encoding/utf-16-with-type-header"] autorelease];
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request startSynchronous];
	success = [request responseEncoding] == NSUnicodeStringEncoding;
	GHAssertTrue(success,[NSString stringWithFormat:@"Failed to parse the content type header correctly"]);
}

- (void)testTimeOut
{
	NSURL *url = [[[NSURL alloc] initWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/the_great_american_novel_%28abridged%29.txt"] autorelease];
	ASIHTTPRequest *request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setTimeOutSeconds:0.0001]; //It's pretty unlikely we will be able to grab the data this quickly, so the request should timeout
	[request startSynchronous];
	
	BOOL success = [[request error] code] == ASIRequestTimedOutErrorType;
	GHAssertTrue(success,@"Timeout didn't generate the correct error");
	
	[ASIHTTPRequest setDefaultTimeOutSeconds:0.0001];
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request startSynchronous];
	
	success = [[request error] code] == ASIRequestTimedOutErrorType;
	GHAssertTrue(success,@"Failed to change the default timeout");	
	
	[ASIHTTPRequest setDefaultTimeOutSeconds:10];
}


// Test fix for a bug that might have caused timeouts when posting data
- (void)testTimeOutWithoutDownloadDelegate
{
	ASIHTTPRequest *request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://trails-network.net/Downloads/MemexTrails_1.0b1.zip"]];
	[request setTimeOutSeconds:5];
	[request setShowAccurateProgress:NO];
	[request setPostBody:[NSMutableData dataWithData:[@"Small Body" dataUsingEncoding:NSUTF8StringEncoding]]];
	[request startSynchronous];
	
	GHAssertNil([request error],@"Generated an error (most likely a timeout) - this test might fail on high latency connections");	
}


- (void)testRequestMethod
{
	NSURL *url = [[[NSURL alloc] initWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/request-method"] autorelease];
	NSArray *methods = [[[NSArray alloc] initWithObjects:@"GET",@"POST",@"PUT",@"DELETE", nil] autorelease];
	for (NSString *method in methods) {
		ASIHTTPRequest *request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
		[request setRequestMethod:method];
		[request startSynchronous];
		BOOL success = [[request responseString] isEqualToString:method];
		GHAssertTrue(success,@"Failed to set the request method correctly");	
	}
	
	// Test to ensure we don't change the request method when we have an unrecognised method already set
	ASIHTTPRequest *request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setRequestMethod:@"FINK"];
	[request appendPostData:[@"King" dataUsingEncoding:NSUTF8StringEncoding]];
	[request buildPostBody];
	BOOL success = [[request requestMethod] isEqualToString:@"FINK"];
	GHAssertTrue(success,@"Erroneously changed request method");	
}

- (void)testHTTPVersion
{
	NSURL *url = [[[NSURL alloc] initWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/http-version"] autorelease];
	ASIHTTPRequest *request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request startSynchronous];
	
	BOOL success = [[request responseString] isEqualToString:@"HTTP/1.1"];
	GHAssertTrue(success,@"Wrong HTTP version used (May fail when using a proxy that changes the HTTP version!)");
	
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseHTTPVersionOne:YES];
	[request startSynchronous];
	
	success = [[request responseString] isEqualToString:@"HTTP/1.0"];
	GHAssertTrue(success,@"Wrong HTTP version used (May fail when using a proxy that changes the HTTP version!)");	
}

- (void)testUserAgent
{
	// defaultUserAgentString will be nil if we haven't set a Bundle Name or Bundle Display Name
	if ([ASIHTTPRequest defaultUserAgentString]) {
		
		// Returns the user agent it received in the response body
		ASIHTTPRequest *request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/user-agent"]];
		[request startSynchronous];
		BOOL success = [[request responseString] isEqualToString:[ASIHTTPRequest defaultUserAgentString]];
		GHAssertTrue(success,@"Failed to set the correct user agent");
	}
	
	// Now test specifying a custom user agent
	ASIHTTPRequest *request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/user-agent"]];
	[request addRequestHeader:@"User-Agent" value:@"Ferdinand Fuzzworth's Magic Tent of Mystery"];
	[request startSynchronous];
	BOOL success = [[request responseString] isEqualToString:@"Ferdinand Fuzzworth's Magic Tent of Mystery"];
	GHAssertTrue(success,@"Failed to set the correct user agent");
	
}

- (void)testAutomaticRedirection
{
	ASIHTTPRequest *request;
	ASIFormDataRequest *request2;
	BOOL success;
	unsigned int i;
	for (i=301; i<308; i++) {
		
		if (i > 304 && i < 307) {
			continue;
		}
		NSURL *url = [NSURL URLWithString:[NSString stringWithFormat:@"http://allseeing-i.com/ASIHTTPRequest/tests/redirect/%hi",i]];
		request = [ASIHTTPRequest requestWithURL:url];
		[request setShouldRedirect:NO];
		[request startSynchronous];
		if (i == 304) { // 304s will not contain a body, as per rfc2616. Will test 304 handling in a future test when we have etag support
			continue;
		}
		success = [[request responseString] isEqualToString:[NSString stringWithFormat:@"Non-redirected content with %hi status code",i]];
		GHAssertTrue(success,[NSString stringWithFormat:@"Got the wrong content when not redirecting after a %hi",i]);
	
		request2 = [ASIFormDataRequest requestWithURL:url];
		[request2 setPostValue:@"Giant Monkey" forKey:@"lookbehindyou"];
		[request2 startSynchronous];
		
		NSString *method = @"GET";
		if (i>304) {
			method = @"POST";	
		}
		NSString *expectedString = [NSString stringWithFormat:@"Redirected as %@ after a %hi status code",method,i];
		if (i>304) {
			expectedString = [NSString stringWithFormat:@"%@\r\nWatch out for the Giant Monkey!",expectedString];
		}

		success = [[request2 responseString] isEqualToString:expectedString];
		GHAssertTrue(success,[NSString stringWithFormat:@"Got the wrong content when redirecting after a %hi",i]);
	
		success = ([request2 responseStatusCode] == 200);
		GHAssertTrue(success,@"Got the wrong status code (expected 200)");

	}
	
	// Test RFC 2616 behaviour
	for (i=301; i<303; i++) {
		
		NSURL *url = [NSURL URLWithString:[NSString stringWithFormat:@"http://allseeing-i.com/ASIHTTPRequest/tests/redirect/%hi",i]];
		request2 = [ASIFormDataRequest requestWithURL:url];
		[request2 setPostValue:@"Giant Monkey" forKey:@"lookbehindyou"];
		[request2 setShouldUseRFC2616RedirectBehaviour:YES];
		[request2 startSynchronous];
		
		success = ([request2 responseStatusCode] == 200);
		GHAssertTrue(success,@"Got the wrong status code (expected 200)");	

		if (i == 303) {
			success = ([request2 postLength] == 0 && ![request2 postBody] && [[request2 requestMethod] isEqualToString:@"GET"]);
			GHAssertTrue(success,@"Failed to reset request to GET on 303 redirect");
			
			success = [[request2 responseString] isEqualToString:[NSString stringWithFormat:@"Redirected as GET after a %hi status code",i]];
			GHAssertTrue(success,@"Failed to dump the post body on 303 redirect");
			
		} else {
			success = ([request2 postLength] > 0 || ![request2 postBody] || ![[request2 requestMethod] isEqualToString:@"POST"]);
			GHAssertTrue(success,@"Failed to use the same request method and body for a redirect when using rfc2616 behaviour");
		
			success = ([[request2 responseString] isEqualToString:[NSString stringWithFormat:@"Redirected as POST after a %hi status code\r\nWatch out for the Giant Monkey!",i]]);
			GHAssertTrue(success,@"Failed to send the correct post body on redirect");
		}
	}
	
	// Ensure the file contains only the body of the last request (after redirects) when downloading to a file
	request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/redirect/301"]];
	NSString *path = [[self filePathForTemporaryTestFiles] stringByAppendingPathComponent:@"test.txt"];
	[request setDownloadDestinationPath:path];
	[request startSynchronous];
	NSString *result = [NSString stringWithContentsOfFile:path encoding:NSUTF8StringEncoding error:nil];
	success = [result isEqualToString:@"Redirected as GET after a 301 status code"];
	GHAssertTrue(success,@"Failed to store just the body of the file request on redirect");
	
	success = ([request originalURL] != [request url]);
	GHAssertTrue(success,@"Failed to update request url on redirection");
	
	success = ([[[request originalURL] absoluteString] isEqualToString:@"http://allseeing-i.com/ASIHTTPRequest/tests/redirect/301"]);
	GHAssertTrue(success,@"Failed to preserve original url");	
}

// Using a persistent connection for HTTP 305-307 would cause crashes on the redirect, not really sure why
// Since 305 (use proxy) wasn't properly supported anyway, 306 is unused, and clients are supposed to confirm redirects for 307, I've simply removed automatic redirect for these codes
- (void)test30xCrash
{
	int i;
	for (i=305; i<308; i++) {
		ASIFormDataRequest *request = [ASIFormDataRequest requestWithURL:[NSURL URLWithString:[NSString stringWithFormat:@"http://allseeing-i.com/ASIHTTPRequest/tests/redirect/%hi",i]]];
		[request setPostValue:@"foo" forKey:@"eep"];
		[request setShouldRedirect:NO];
		[request startSynchronous];
		request = [ASIFormDataRequest requestWithURL:[NSURL URLWithString:[NSString stringWithFormat:@"http://allseeing-i.com/ASIHTTPRequest/tests/redirect/%hi",i]]];
		[request setPostValue:@"foo" forKey:@"eep"];
		[request startSynchronous];
	}
}

- (void)testUploadContentLength
{
	//This url will return the contents of the Content-Length request header
	NSURL *url = [NSURL URLWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/content-length"];
	ASIHTTPRequest *request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setPostBody:[NSMutableData dataWithLength:1024*32]];
	[request startSynchronous];
	
	BOOL success = ([[request responseString] isEqualToString:[NSString stringWithFormat:@"%hu",(1024*32)]]);
	GHAssertTrue(success,@"Sent wrong content length");
}

- (void)testDownloadContentLength
{
	NSURL *url = [[[NSURL alloc] initWithString:@"http://allseeing-i.com/i/logo.png"] autorelease];
	ASIHTTPRequest *request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request startSynchronous];
	
	BOOL success = ([request contentLength] == 18443);
	GHAssertTrue(success,@"Got wrong content length");
}

- (void)testFileDownload
{
	NSString *path = [[self filePathForTemporaryTestFiles] stringByAppendingPathComponent:@"testimage.png"];
	
	NSURL *url = [[[NSURL alloc] initWithString:@"http://allseeing-i.com/i/logo.png"] autorelease];
	ASIHTTPRequest *request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setDownloadDestinationPath:path];
	[request startSynchronous];

#if TARGET_OS_IPHONE
	UIImage *image = [[[UIImage alloc] initWithContentsOfFile:path] autorelease];
#else
	NSImage *image = [[[NSImage alloc] initWithContentsOfFile:path] autorelease];	
#endif
	
	GHAssertNotNil(image,@"Failed to download data to a file");
}


- (void)testCompressedResponseDownloadToFile
{
	NSString *path = [[self filePathForTemporaryTestFiles] stringByAppendingPathComponent:@"testfile"];
	
	NSURL *url = [[[NSURL alloc] initWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/first"] autorelease];
	ASIHTTPRequest *request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setDownloadDestinationPath:path];
	[request startSynchronous];
	
	NSString *tempPath = [request temporaryFileDownloadPath];
	GHAssertNil(tempPath,@"Failed to clean up temporary download file");		
	
	//BOOL success = (![[NSFileManager defaultManager] fileExistsAtPath:tempPath]);
	//GHAssertTrue(success,@"Failed to remove file from temporary location");	
	
	BOOL success = [[NSString stringWithContentsOfURL:[NSURL fileURLWithPath:path] encoding:NSUTF8StringEncoding error:NULL] isEqualToString:@"This is the expected content for the first string"];
	GHAssertTrue(success,@"Failed to download data to a file");
	
}


- (void)testDownloadProgress
{
	// We run tests that measure progress on the main thread because otherwise we can't depend on the progress delegate being notified before we need to test it's working
	[self performSelectorOnMainThread:@selector(performDownloadProgressTest) withObject:nil waitUntilDone:YES];
}

- (void)performDownloadProgressTest
{
	progress = 0;
	NSURL *url = [[[NSURL alloc] initWithString:@"http://allseeing-i.com/i/logo.png"] autorelease];
	ASIHTTPRequest *request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setDownloadProgressDelegate:self];
	[request startSynchronous];

	BOOL success = (progress == 1.0);
	GHAssertTrue(success,@"Failed to properly increment download progress %f != 1.0",progress);	
	
	progress = 0;
	request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/the_great_american_novel.txt"]];
	[request setDownloadProgressDelegate:self];
	[request startAsynchronous];
	
	[[NSRunLoop currentRunLoop] runUntilDate:[NSDate dateWithTimeIntervalSinceNow:2]];
	
	success = (progress != 1.0);
	GHAssertTrue(success,@"Downloaded too quickly, cannot proceed with test");	
	 
	success = (progress > 0);
	GHAssertTrue(success,@"Either downloaded too slowly, or progress is not being correctly updated");		 
	
	
}

- (void)testUploadProgress
{
	// We run tests that measure progress on the main thread because otherwise we can't depend on the progress delegate being notified before we need to test it's working
	[self performSelectorOnMainThread:@selector(performUploadProgressTest) withObject:nil waitUntilDone:YES];	
}

- (void)performUploadProgressTest
{
	progress = 0;
	ASIHTTPRequest *request = [[[ASIHTTPRequest alloc] initWithURL:[NSURL URLWithString:@"http://allseeing-i.com/ignore"]] autorelease];
	[request setPostBody:(NSMutableData *)[@"This is the request body" dataUsingEncoding:NSUTF8StringEncoding]];
	[request setUploadProgressDelegate:self];
	[request startSynchronous];
	
	
	BOOL success = (progress == 1.0);
	GHAssertTrue(success,@"Failed to properly increment upload progress %f != 1.0",progress);	
}

- (void)testPostBodyStreamedFromDisk
{
	// We run tests that measure progress on the main thread because otherwise we can't depend on the progress delegate being notified before we need to test it's working
	[self performSelectorOnMainThread:@selector(performPostBodyStreamedFromDiskTest) withObject:nil waitUntilDone:YES];
	
}

- (void)performPostBodyStreamedFromDiskTest
{
	NSURL *url = [NSURL URLWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/print_request_body"];
	NSString *requestBody = @"This is the request body";
	NSString *requestContentPath = [[self filePathForTemporaryTestFiles] stringByAppendingPathComponent:@"testfile.txt"];
	[[requestBody dataUsingEncoding:NSUTF8StringEncoding] writeToFile:requestContentPath atomically:NO];
	
	
	// Test using a user-specified file as the request body (useful for PUT)
	progress = 0;
	ASIHTTPRequest *request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setRequestMethod:@"PUT"];
	[request setShouldStreamPostDataFromDisk:YES];
	[request setUploadProgressDelegate:self];
	[request setPostBodyFilePath:requestContentPath];
	[request startSynchronous];
	
	BOOL success = (progress == 1.0);
	GHAssertTrue(success,@"Failed to properly increment upload progress %f != 1.0",progress);
	
	success = [[request responseString] isEqualToString:requestBody];
	GHAssertTrue(success,@"Failed upload the correct request body");
	
	
	// Test building a request body by appending data
	progress = 0;
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setShouldStreamPostDataFromDisk:YES];
	[request setRequestMethod:@"PUT"];
	[request setUploadProgressDelegate:self];
	[request appendPostDataFromFile:requestContentPath];
	[request startSynchronous];
	
	success = (progress == 1.0);
	GHAssertTrue(success,@"Failed to properly increment upload progress %f != 1.0",progress);
	
	success = [[request responseString] isEqualToString:requestBody];
	GHAssertTrue(success,@"Failed upload the correct request body");		
}



- (void)testCookies
{
	BOOL success;
	
	// Set setting a cookie
	NSURL *url = [[[NSURL alloc] initWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/set_cookie"] autorelease];
	ASIHTTPRequest *request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseCookiePersistance:YES];
	[request startSynchronous];
	NSString *html = [request responseString];
	success = [html isEqualToString:@"I have set a cookie"];
	GHAssertTrue(success,@"Failed to set a cookie");
	
	// Test a cookie is stored in responseCookies
	NSArray *cookies = [request responseCookies];
	GHAssertNotNil(cookies,@"Failed to store cookie data in responseCookies");
	

	// Test the cookie contains the correct data
	NSHTTPCookie *cookie = nil;
	BOOL foundCookie = NO;
	for (cookie in cookies) {
		if ([[cookie name] isEqualToString:@"ASIHTTPRequestTestCookie"]) {
			foundCookie = YES;
			success = [[[cookie value] decodedCookieValue] isEqualToString:@"This is the value"];
			GHAssertTrue(success,@"Failed to store the correct value for a cookie");
			success = [[cookie domain] isEqualToString:@"allseeing-i.com"];
			GHAssertTrue(success,@"Failed to store the correct domain for a cookie");
			success = [[cookie path] isEqualToString:@"/ASIHTTPRequest/tests"];
			GHAssertTrue(success,@"Failed to store the correct path for a cookie");
			break;
		}
	}
	GHAssertTrue(foundCookie,@"Failed store a particular cookie - can't continue with the rest of the tests");
	
	if (!foundCookie) {
		return;
	}
	// Test a cookie is presented when manually added to the request
	url = [[[NSURL alloc] initWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/read_cookie"] autorelease];
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseCookiePersistance:NO];
	[request setRequestCookies:[NSMutableArray arrayWithObject:cookie]];
	[request startSynchronous];
	html = [request responseString];
	success = [html isEqualToString:@"I have 'This is the value' as the value of 'ASIHTTPRequestTestCookie'"];
	GHAssertTrue(success,@"Cookie not presented to the server with cookie persistance OFF");

	// Test a cookie is presented from the persistent store
	url = [[[NSURL alloc] initWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/read_cookie"] autorelease];
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseCookiePersistance:YES];
	[request startSynchronous];
	html = [request responseString];
	success = [html isEqualToString:@"I have 'This is the value' as the value of 'ASIHTTPRequestTestCookie'"];
	GHAssertTrue(success,@"Cookie not presented to the server with cookie persistance ON");
	
	// Test removing a cookie
	url = [[[NSURL alloc] initWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/remove_cookie"] autorelease];
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request startSynchronous];
	html = [request responseString];
	success = [html isEqualToString:@"I have removed a cookie"];
	GHAssertTrue(success,@"Failed to remove a cookie");

	// Test making sure cookie was properly removed
	url = [[[NSURL alloc] initWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/read_cookie"] autorelease];
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request startSynchronous];
	html = [request responseString];
	success = [html isEqualToString:@"No cookie exists"];
	GHAssertTrue(success,@"Cookie presented to the server when it should have been removed");
	
	// Test setting a custom cookie works
	NSDictionary *cookieProperties = [[[NSMutableDictionary alloc] init] autorelease];
	
	// We'll add a line break to our cookie value to test it gets correctly encoded
	[cookieProperties setValue:[@"Test\r\nValue" encodedCookieValue] forKey:NSHTTPCookieValue];
	[cookieProperties setValue:@"ASIHTTPRequestTestCookie" forKey:NSHTTPCookieName];
	[cookieProperties setValue:@"allseeing-i.com" forKey:NSHTTPCookieDomain];
	[cookieProperties setValue:[NSDate dateWithTimeIntervalSinceNow:60*60*4] forKey:NSHTTPCookieExpires];
	[cookieProperties setValue:@"/ASIHTTPRequest/tests" forKey:NSHTTPCookiePath];
	cookie = [[[NSHTTPCookie alloc] initWithProperties:cookieProperties] autorelease];
	
	GHAssertNotNil(cookie,@"Failed to create a cookie - cookie value was not correctly encoded?");

	url = [[[NSURL alloc] initWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/read_cookie"] autorelease];
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseCookiePersistance:NO];
	[request setRequestCookies:[NSMutableArray arrayWithObject:cookie]];
	[request startSynchronous];
	html = [request responseString];
	success = [html isEqualToString:@"I have 'Test\r\nValue' as the value of 'ASIHTTPRequestTestCookie'"];
	GHAssertTrue(success,@"Custom cookie not presented to the server with cookie persistance OFF");
	

	// Test removing all cookies works
	[ASIHTTPRequest clearSession];
	NSArray *sessionCookies = [ASIHTTPRequest sessionCookies];
	success = ([sessionCookies count] == 0);
	GHAssertTrue(success,@"Cookies not removed");

	url = [[[NSURL alloc] initWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/read_cookie"] autorelease];
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseCookiePersistance:YES];
	[request startSynchronous];
	html = [request responseString];
	success = [html isEqualToString:@"No cookie exists"];
	GHAssertTrue(success,@"Cookie presented to the server when it should have been removed");
	
	// Test fetching cookies for a relative url - fixes a problem where urls created with URLWithString:relativeToURL: wouldn't always read cookies from the persistent store
	[ASIHTTPRequest clearSession];
	
	url = [[[NSURL alloc] initWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/set_cookie"] autorelease];
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseCookiePersistance:YES];
	[request setUseSessionPersistance:NO];
	[request startSynchronous];
	
	NSURL *originalURL = [NSURL URLWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/"];
	url = [NSURL URLWithString:@"read_cookie" relativeToURL:originalURL];
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseCookiePersistance:YES];
	[request setUseSessionPersistance:NO];
	[request startSynchronous];
	html = [request responseString];
	NSLog(@"%@",html);
	success = [html isEqualToString:@"I have 'This is the value' as the value of 'ASIHTTPRequestTestCookie'"];
	GHAssertTrue(success,@"Custom cookie not presented to the server with cookie persistance OFF");
	
}

// Test fix for a crash if you tried to remove credentials that didn't exist
- (void)testRemoveCredentialsFromKeychain
{
	[ASIHTTPRequest removeCredentialsForHost:@"apple.com" port:0 protocol:@"http" realm:@"Nothing to see here"];
	[ASIHTTPRequest removeCredentialsForProxy:@"apple.com" port:0 realm:@"Nothing to see here"];
	
}


- (void)testBasicAuthentication
{
	[ASIHTTPRequest removeCredentialsForHost:@"allseeing-i.com" port:0 protocol:@"http" realm:@"SECRET_STUFF"];
	[ASIHTTPRequest clearSession];

	NSURL *url = [NSURL URLWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/basic-authentication"];
	ASIHTTPRequest *request;
	BOOL success;
	NSError *err;
	
	// Test authentication needed when no credentials supplied
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseKeychainPersistance:NO];
	[request startSynchronous];
	success = [[request error] code] == ASIAuthenticationErrorType;
	GHAssertTrue(success,@"Failed to generate permission denied error with no credentials");
	
	// Test wrong credentials supplied
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseKeychainPersistance:NO];
	[request setUsername:@"wrong"];
	[request setPassword:@"wrong"];
	[request startSynchronous];
	success = [[request error] code] == ASIAuthenticationErrorType;
	GHAssertTrue(success,@"Failed to generate permission denied error with wrong credentials");
	
	// Test correct credentials supplied
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseSessionPersistance:YES];
	[request setUseKeychainPersistance:YES];
	[request setShouldPresentCredentialsBeforeChallenge:NO];
	[request setUsername:@"secret_username"];
	[request setPassword:@"secret_password"];
	[request startSynchronous];
	err = [request error];
	GHAssertNil(err,@"Failed to supply correct username and password");
	
	// Ensure credentials are not reused
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseSessionPersistance:NO];
	[request setUseKeychainPersistance:NO];
	[request startSynchronous];
	success = [[request error] code] == ASIAuthenticationErrorType;
	GHAssertTrue(success,@"Reused credentials when we shouldn't have");

	// Ensure credentials stored in the session are reused
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseSessionPersistance:YES];
	[request setUseKeychainPersistance:NO];
	[request startSynchronous];
	err = [request error];
	GHAssertNil(err,@"Failed to reuse credentials");
	
	[ASIHTTPRequest clearSession];
	
	// Ensure credentials stored in the session were wiped
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseKeychainPersistance:NO];
	[request startSynchronous];
	success = [[request error] code] == ASIAuthenticationErrorType;
	GHAssertTrue(success,@"Failed to clear credentials");
	
	// Ensure credentials stored in the keychain are reused
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseKeychainPersistance:YES];
	[request startSynchronous];
	err = [request error];
	GHAssertNil(err,@"Failed to use stored credentials");
	
	[ASIHTTPRequest removeCredentialsForHost:@"allseeing-i.com" port:0 protocol:@"http" realm:@"SECRET_STUFF"];
	
	// Ensure credentials stored in the keychain were wiped
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseKeychainPersistance:YES];
	[request setUseSessionPersistance:NO];
	[request startSynchronous];
	success = [[request error] code] == ASIAuthenticationErrorType;
	GHAssertTrue(success,@"Failed to clear credentials");
	
	// Tests shouldPresentCredentialsBeforeChallenge with credentials stored in the session
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseSessionPersistance:YES];
	[request startSynchronous];
	success = [request authenticationRetryCount] == 0;
	GHAssertTrue(success,@"Didn't supply credentials before being asked for them when talking to the same server with shouldPresentCredentialsBeforeChallenge == YES");	
	
	// Ensure credentials stored in the session were not presented to the server before it asked for them
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseSessionPersistance:YES];
	[request setShouldPresentCredentialsBeforeChallenge:NO];
	[request startSynchronous];
	success = [request authenticationRetryCount] == 1;
	GHAssertTrue(success,@"Supplied session credentials before being asked for them");	
	
	[ASIHTTPRequest clearSession];
	
	// Test credentials set on the request are sent before the server asks for them
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseSessionPersistance:NO];
	[request setUsername:@"secret_username"];
	[request setPassword:@"secret_password"];
	[request setShouldPresentCredentialsBeforeChallenge:YES];
	[request startSynchronous];
	success = [request authenticationRetryCount] == 0;
	GHAssertTrue(success,@"Didn't supply credentials before being asked for them, even though they were set on the request and shouldPresentCredentialsBeforeChallenge == YES");	
	
	// Test credentials set on the request aren't sent before the server asks for them
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseSessionPersistance:NO];
	[request setUsername:@"secret_username"];
	[request setPassword:@"secret_password"];
	[request setShouldPresentCredentialsBeforeChallenge:NO];
	[request startSynchronous];
	success = [request authenticationRetryCount] == 1;
	GHAssertTrue(success,@"Supplied request credentials before being asked for them");	
	
	
	// Test credentials presented before a challenge are stored in the session store
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUsername:@"secret_username"];
	[request setPassword:@"secret_password"];
	[request setShouldPresentCredentialsBeforeChallenge:YES];
	[request startSynchronous];
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request startSynchronous];
	err = [request error];
	GHAssertNil(err,@"Failed to use stored credentials");	
	
	
	// Ok, now let's test on a different server to sanity check that the credentials from out previous requests are not being used
	url = [NSURL URLWithString:@"https://selfsigned.allseeing-i.com/ASIHTTPRequest/tests/basic-authentication"];
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseSessionPersistance:YES];
	[request setUseKeychainPersistance:NO];
	[request setValidatesSecureCertificate:NO];
	[request startSynchronous];
	success = [[request error] code] == ASIAuthenticationErrorType;
	GHAssertTrue(success,@"Reused credentials when we shouldn't have");	
	
}



- (void)testDigestAuthentication
{
	[ASIHTTPRequest removeCredentialsForHost:@"allseeing-i.com" port:0 protocol:@"http" realm:@"Keep out"];
	[ASIHTTPRequest clearSession];
	
	NSURL *url = [[[NSURL alloc] initWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/digest-authentication"] autorelease];
	ASIHTTPRequest *request;
	BOOL success;
	NSError *err;
	
	// Test authentication needed when no credentials supplied
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseKeychainPersistance:NO];
	[request startSynchronous];
	success = [[request error] code] == ASIAuthenticationErrorType;
	GHAssertTrue(success,@"Failed to generate permission denied error with no credentials");
	
	// Test wrong credentials supplied
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseKeychainPersistance:NO];
	[request setUsername:@"wrong"];
	[request setPassword:@"wrong"];
	[request startSynchronous];
	success = [[request error] code] == ASIAuthenticationErrorType;
	GHAssertTrue(success,@"Failed to generate permission denied error with wrong credentials");
	
	// Test correct credentials supplied
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseSessionPersistance:YES];
	[request setUseKeychainPersistance:YES];
	[request setUsername:@"secret_username"];
	[request setPassword:@"secret_password"];
	[request startSynchronous];
	err = [request error];
	GHAssertNil(err,@"Failed to supply correct username and password");
	
	// Ensure credentials are not reused
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseSessionPersistance:NO];
	[request setUseKeychainPersistance:NO];
	[request startSynchronous];
	success = [[request error] code] == ASIAuthenticationErrorType;
	GHAssertTrue(success,@"Reused credentials when we shouldn't have");
	
	// Ensure credentials stored in the session are reused
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseSessionPersistance:YES];
	[request setUseKeychainPersistance:NO];
	[request startSynchronous];
	err = [request error];
	GHAssertNil(err,@"Failed to reuse credentials");
	
	[ASIHTTPRequest clearSession];
	
	// Ensure credentials stored in the session were wiped
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseKeychainPersistance:NO];
	[request startSynchronous];
	success = [[request error] code] == ASIAuthenticationErrorType;
	GHAssertTrue(success,@"Failed to clear credentials");
	
	// Ensure credentials stored in the keychain are reused
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseKeychainPersistance:YES];
	[request startSynchronous];
	err = [request error];
	GHAssertNil(err,@"Failed to reuse credentials");
	
	[ASIHTTPRequest removeCredentialsForHost:@"allseeing-i.com" port:0 protocol:@"http" realm:@"Keep out"];
	
	// Ensure credentials stored in the keychain were wiped
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseKeychainPersistance:YES];
	[request setUseSessionPersistance:NO];
	[request startSynchronous];
	success = [[request error] code] == ASIAuthenticationErrorType;
	GHAssertTrue(success,@"Failed to clear credentials");	
	
	
	// Test credentials set on the request are sent before the server asks for them
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setUseSessionPersistance:YES];
	[request setShouldPresentCredentialsBeforeChallenge:YES];
	[request startSynchronous];
	success = [request authenticationRetryCount] == 0;
	GHAssertTrue(success,@"Didn't supply credentials before being asked for them, even though they were set in the session and shouldPresentCredentialsBeforeChallenge == YES");	
	
}

- (void)testNTLMHandshake
{
	// This test connects to a script that masquerades as an NTLM server
	// It tests that the handshake seems sane, but doesn't actually authenticate
	
	[ASIHTTPRequest clearSession];
	
	NSURL *url = [NSURL URLWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/pretend-ntlm-handshake"];
	
	ASIHTTPRequest *request = [ASIHTTPRequest requestWithURL:url];
	[request setUseKeychainPersistance:NO];
	[request setUseSessionPersistance:NO];
	[request startSynchronous];
	BOOL success = [[request error] code] == ASIAuthenticationErrorType;
	GHAssertTrue(success,@"Failed to generate permission denied error with no credentials");


	request = [ASIHTTPRequest requestWithURL:url];
	[request setUseSessionPersistance:YES];
	[request setUseKeychainPersistance:NO];
	[request setUsername:@"king"];
	[request setPassword:@"fink"];
	[request setDomain:@"Castle.Kingdom"];
	[request startSynchronous];

	GHAssertNil([request error],@"Got an error when credentials were supplied");
	
	// NSProcessInfo returns a lower case string for host name, while CFNetwork will send a mixed case string for host name, so we'll compare by lowercasing everything
	NSString *hostName = [[NSProcessInfo processInfo] hostName];
	NSString *expectedResponse = [[NSString stringWithFormat:@"You are %@ from %@/%@",@"king",@"Castle.Kingdom",hostName] lowercaseString];
	success = [[[request responseString] lowercaseString] isEqualToString:expectedResponse];
	GHAssertTrue(success,@"Failed to send credentials correctly? (Expected: '%@', got '%@')",expectedResponse,[[request responseString] lowercaseString]);
}

- (void)testCompressedResponse
{
	// allseeing-i.com does not gzip png images
	NSURL *url = [[[NSURL alloc] initWithString:@"http://allseeing-i.com/i/logo.png"] autorelease];
	ASIHTTPRequest *request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request startSynchronous];
	NSString *encoding = [[request responseHeaders] objectForKey:@"Content-Encoding"];
	BOOL success = (!encoding || [encoding rangeOfString:@"gzip"].location != NSNotFound);
	GHAssertTrue(success,@"Got incorrect request headers from server");
	
	success = ([request rawResponseData] == [request responseData]);
	GHAssertTrue(success,@"Attempted to uncompress data that was not compressed");	
	
	url = [[[NSURL alloc] initWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/first"] autorelease];
	request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request startSynchronous];
	success = ([request rawResponseData] != [request responseData]);
	GHAssertTrue(success,@"Uncompressed data is the same as compressed data");	
	
	success = [[request responseString] isEqualToString:@"This is the expected content for the first string"];
	GHAssertTrue(success,@"Failed to decompress data correctly?");
}


- (void)testPartialFetch
{
	// We run tests that measure progress on the main thread because otherwise we can't depend on the progress delegate being notified before we need to test it's working
	[self performSelectorOnMainThread:@selector(performPartialFetchTest) withObject:nil waitUntilDone:YES];	

}
					
- (void)performPartialFetchTest
{
	NSString *downloadPath = [[self filePathForTemporaryTestFiles] stringByAppendingPathComponent:@"testfile.txt"];
	NSString *tempPath = [[self filePathForTemporaryTestFiles] stringByAppendingPathComponent:@"tempfile.txt"];
	NSString *partialContent = @"This file should be exactly 163 bytes long when encoded as UTF8, Unix line breaks with no BOM.\n";
	[partialContent writeToFile:tempPath atomically:NO encoding:NSASCIIStringEncoding error:nil];
	
	progress = 0;
	NSURL *url = [[[NSURL alloc] initWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/test_partial_download.txt"] autorelease];
	ASIHTTPRequest *request = [[[ASIHTTPRequest alloc] initWithURL:url] autorelease];
	[request setDownloadDestinationPath:downloadPath];
	[request setTemporaryFileDownloadPath:tempPath];
	[request setAllowResumeForFileDownloads:YES];
	[request setDownloadProgressDelegate:self];
	[request startSynchronous];
	
	BOOL success = ([request contentLength] == 68);
	GHAssertTrue(success,@"Failed to download a segment of the data");
	
	NSString *content = [NSString stringWithContentsOfFile:downloadPath encoding:NSUTF8StringEncoding error:NULL];
	
	NSString *newPartialContent = [content substringFromIndex:95];
	success = ([newPartialContent isEqualToString:@"This is the content we ought to be getting if we start from byte 95."]);
	GHAssertTrue(success,@"Failed to append the correct data to the end of the file?");
	
	success = (progress == 1.0);
	GHAssertTrue(success,@"Failed to correctly display increment progress for a partial download");
}

// The '000' is to ensure this test runs first, as another test may connect to https://selfsigned.allseeing-i.com and accept the certificate
- (void)test000SSL
{
	NSURL *url = [NSURL URLWithString:@"https://selfsigned.allseeing-i.com"];
	ASIHTTPRequest *request = [ASIHTTPRequest requestWithURL:url];
	[request startSynchronous];
	
	GHAssertNotNil([request error],@"Failed to generate an error for a self-signed certificate (Will fail on the second run in the same session!)");		
	
	// Just for testing the request generated a custom error description - don't do this! You should look at the domain / code of the underlyingError in your own programs.
	BOOL success = ([[[request error] localizedDescription] isEqualToString:@"A connection failure occurred: SSL problem (possibily a bad/expired/self-signed certificate)"]);
	GHAssertTrue(success,@"Generated the wrong error for a self signed cert");
	
	// Turn off certificate validation, and try again
	request = [ASIHTTPRequest requestWithURL:url];
	[request setValidatesSecureCertificate:NO];
	[request startSynchronous];
	
	GHAssertNil([request error],@"Failed to accept a self-signed certificate");	
}

- (void)testRedirectPreservesSession
{
	// Remove any old session cookies
	[ASIHTTPRequest clearSession];
	ASIHTTPRequest *request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/session_redirect"]];
	[request startSynchronous];
	BOOL success = [[request responseString] isEqualToString:@"Take me to your leader"];
	GHAssertTrue(success,@"Failed to redirect preserving session cookies");	
}

- (void)testTooMuchRedirection
{
	// This url will simply send a 302 redirect back to itself
	ASIHTTPRequest *request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/one_infinite_loop"]];
	[request startSynchronous];
	GHAssertNotNil([request error],@"Failed to generate an error when redirection occurs too many times");
	BOOL success = ([[request error] code] == ASITooMuchRedirectionErrorType);
	GHAssertTrue(success,@"Generated the wrong error for a redirection loop");		
}

- (void)testRedirectToNewDomain
{
	ASIHTTPRequest *request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/redirect_to_new_domain"]];
	[request startSynchronous];
	BOOL success = [[[request url] absoluteString] isEqualToString:@"http://www.apple.com/"];
	GHAssertTrue(success,@"Failed to redirect to a different domain");		
}

// Ensure request method changes to get
- (void)test303Redirect
{
	ASIHTTPRequest *request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/redirect_303"]];
	[request setRequestMethod:@"PUT"];
	[request appendPostData:[@"Fuzzy" dataUsingEncoding:NSUTF8StringEncoding]];
	[request startSynchronous];
	BOOL success = [[[request url] absoluteString] isEqualToString:@"http://allseeing-i.com/ASIHTTPRequest/tests/request-method"];
	GHAssertTrue(success,@"Failed to redirect to correct location");
	success = [[request responseString] isEqualToString:@"GET"];
	GHAssertTrue(success,@"Failed to use GET on new URL");
}

- (void)testCompression
{
	NSString *content = @"This is the test content. This is the test content. This is the test content. This is the test content.";
	
	// Test in memory compression / decompression
	NSData *data = [content dataUsingEncoding:NSUTF8StringEncoding];
	NSData *compressedData = [ASIHTTPRequest compressData:data];
	NSData *uncompressedData = [ASIHTTPRequest uncompressZippedData:compressedData];
	NSString *newContent = [[[NSString alloc] initWithBytes:[uncompressedData bytes] length:[uncompressedData length] encoding:NSUTF8StringEncoding] autorelease];
	
	BOOL success = [newContent isEqualToString:content];
	GHAssertTrue(success,@"Failed compress or decompress the correct data");	
	
	// Test file to file compression / decompression
	
	NSString *basePath = [self filePathForTemporaryTestFiles];
	NSString *sourcePath = [basePath stringByAppendingPathComponent:@"text.txt"];
	NSString *destPath = [basePath stringByAppendingPathComponent:@"text.txt.compressed"];
	NSString *newPath = [basePath stringByAppendingPathComponent:@"text2.txt"];
	
	[content writeToFile:sourcePath atomically:NO encoding:NSUTF8StringEncoding error:NULL];
	[ASIHTTPRequest compressDataFromFile:sourcePath toFile:destPath];
	[ASIHTTPRequest uncompressZippedDataFromFile:destPath toFile:newPath];
	success = [[NSString stringWithContentsOfFile:newPath encoding:NSUTF8StringEncoding error:NULL] isEqualToString:content];
	GHAssertTrue(success,@"Failed compress or decompress the correct data");
	
	// Test compressed body
	// Body is deflated by ASIHTTPRequest, sent, inflated by the server, printed, deflated by mod_deflate, response is inflated by ASIHTTPRequest
	ASIHTTPRequest *request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/compressed_post_body"]];
	[request setRequestMethod:@"PUT"];
	[request setShouldCompressRequestBody:YES];
	[request appendPostData:data];
	[request startSynchronous];
	
	success = [[request responseString] isEqualToString:content];
	GHAssertTrue(success,@"Failed to compress the body, or server failed to decompress it");	
	
	request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/compressed_post_body"]];
	[request setRequestMethod:@"PUT"];
	[request setShouldCompressRequestBody:YES];
	[request setShouldStreamPostDataFromDisk:YES];
	[request setUploadProgressDelegate:self];
	[request setPostBodyFilePath:sourcePath];
	[request startSynchronous];

	success = [[request responseString] isEqualToString:content];
	GHAssertTrue(success,@"Failed to compress the body, or server failed to decompress it");		
	
}


// Ensure class convenience constructor returns an instance of our subclass
- (void)testSubclass
{
	ASIHTTPRequestSubclass *instance = [ASIHTTPRequestSubclass requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com"]];
	BOOL success = [instance isKindOfClass:[ASIHTTPRequestSubclass class]];
	GHAssertTrue(success,@"Convenience constructor failed to return an instance of the correct class");	
}


- (void)testThrottlingDownloadBandwidth
{
	[ASIHTTPRequest setMaxBandwidthPerSecond:0];
	
	// This content is around 128KB in size, and it won't be gzipped, so it should take more than 8 seconds to download at 14.5KB / second
	// We'll test first without throttling
	ASIHTTPRequest *request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/the_great_american_novel_%28abridged%29.txt"]];
	NSDate *date = [NSDate date];
	[request startSynchronous];	
	
	NSTimeInterval interval =[date timeIntervalSinceNow];
	BOOL success = (interval > -7);
	GHAssertTrue(success,@"Downloaded the file too slowly - either this is a bug, or your internet connection is too slow to run this test (must be able to download 128KB in less than 7 seconds, without throttling)");
	
	// Now we'll test with throttling
	[ASIHTTPRequest setMaxBandwidthPerSecond:ASIWWANBandwidthThrottleAmount];
	request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/the_great_american_novel_%28abridged%29.txt"]];
	date = [NSDate date];
	[request startSynchronous];	
	
	[ASIHTTPRequest setMaxBandwidthPerSecond:0];
	
	interval =[date timeIntervalSinceNow];
	success = (interval < -7);
	GHAssertTrue(success,@"Failed to throttle download");		
	GHAssertNil([request error],@"Request generated an error - timeout?");	
	
}

- (void)testThrottlingUploadBandwidth
{
	[ASIHTTPRequest setMaxBandwidthPerSecond:0];
	
	// Create a 64KB request body
	NSData *data = [[[NSMutableData alloc] initWithLength:64*1024] autorelease];
	
	// We'll test first without throttling
	ASIHTTPRequest *request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com/ignore"]];
	[request appendPostData:data];
	NSDate *date = [NSDate date];
	[request startSynchronous];	
	
	NSTimeInterval interval =[date timeIntervalSinceNow];
	BOOL success = (interval > -3);
	GHAssertTrue(success,@"Uploaded the data too slowly - either this is a bug, or your internet connection is too slow to run this test (must be able to upload 64KB in less than 3 seconds, without throttling)");
	
	// Now we'll test with throttling
	[ASIHTTPRequest setMaxBandwidthPerSecond:ASIWWANBandwidthThrottleAmount];
	request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com/ignore"]];
	[request appendPostData:data];
	date = [NSDate date];
	[request startSynchronous];	
	
	[ASIHTTPRequest setMaxBandwidthPerSecond:0];
	
	interval =[date timeIntervalSinceNow];
	success = (interval < -3);
	GHAssertTrue(success,@"Failed to throttle upload");		
	GHAssertNil([request error],@"Request generated an error - timeout?");	
}


- (void)testFetchToInvalidPath
{
	// Test gzipped content
	ASIHTTPRequest *request = [ASIHTTPRequest requestWithURL:[NSURL	URLWithString:@"http://allseeing-i.com"]];
	[request setDownloadDestinationPath:@"/an/invalid/location.html"];
	[request startSynchronous];
	GHAssertNotNil([request error],@"Failed to generate an authentication when attempting to write to an invalid location");	
	
	//Test non-gzipped content
	request = [ASIHTTPRequest requestWithURL:[NSURL	URLWithString:@"http://allseeing-i.com/i/logo.png"]];
	[request setDownloadDestinationPath:@"/an/invalid/location.png"];
	[request startSynchronous];
	GHAssertNotNil([request error],@"Failed to generate an authentication when attempting to write to an invalid location");		
}

- (void)testResponseStatusMessage
{
	ASIHTTPRequest *request = [ASIHTTPRequest requestWithURL:[NSURL	URLWithString:@"http://allseeing-i.com/the-meaning-of-life"]];
	[request startSynchronous];	
	BOOL success = [[request responseStatusMessage] isEqualToString:@"HTTP/1.0 404 Not Found"];
	GHAssertTrue(success,@"Got wrong response status message");
}

- (void)testAsynchronous
{
	ASIHTTPRequest *request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/first"]];
	[request setUserInfo:[NSDictionary dictionaryWithObject:[NSNumber numberWithInt:1] forKey:@"RequestNumber"]];
	[request setDidFailSelector:@selector(asyncFail:)];
	[request setDidFinishSelector:@selector(asyncSuccess:)];
	[request setDelegate:self];
	[request startAsynchronous];
	
	request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/second"]];
	[request setUserInfo:[NSDictionary dictionaryWithObject:[NSNumber numberWithInt:2] forKey:@"RequestNumber"]];
	[request setDidFailSelector:@selector(asyncFail:)];
	[request setDidFinishSelector:@selector(asyncSuccess:)];
	[request setDelegate:self];
	[request startAsynchronous];
	
	request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/third"]];
	[request setUserInfo:[NSDictionary dictionaryWithObject:[NSNumber numberWithInt:3] forKey:@"RequestNumber"]];
	[request setDidFailSelector:@selector(asyncFail:)];
	[request setDidFinishSelector:@selector(asyncSuccess:)];
	[request setDelegate:self];
	[request startAsynchronous];	
	
	request = [ASIHTTPRequest requestWithURL:nil];
	[request setUserInfo:[NSDictionary dictionaryWithObject:[NSNumber numberWithInt:4] forKey:@"RequestNumber"]];
	[request setDidFailSelector:@selector(asyncFail:)];
	[request setDidFinishSelector:@selector(asyncSuccess:)];
	[request setDelegate:self];
	[request startAsynchronous];	
}


- (void)asyncFail:(ASIHTTPRequest *)request
{
	int requestNumber = [[[request userInfo] objectForKey:@"RequestNumber"] intValue];
	GHAssertEquals(requestNumber,4,@"Wrong request failed");	
}

- (void)asyncSuccess:(ASIHTTPRequest *)request
{
	int requestNumber = [[[request userInfo] objectForKey:@"RequestNumber"] intValue];
	GHAssertNotEquals(requestNumber,4,@"Request succeeded when it should have failed");
	
	BOOL success;
	switch (requestNumber) {
		case 1:
			success = [[request responseString] isEqualToString:@"This is the expected content for the first string"];
			break;
		case 2:
			success = [[request responseString] isEqualToString:@"This is the expected content for the second string"];
			break;
		case 3:
			success = [[request responseString] isEqualToString:@"This is the expected content for the third string"];
			break;
	}
	GHAssertTrue(success,@"Got wrong request content - very bad!");
	
}

- (void)testCloseConnection
{
	ASIHTTPRequest *request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com/ASIHTTPRequest/tests/close-connection"]];
	[request startSynchronous];
	
	BOOL success = ![request connectionCanBeReused];
	GHAssertTrue(success,@"Should not be able to re-use a request sent with Connection:close");
	
}


// Will be called on Mac OS
- (void)setDoubleValue:(double)newProgress;
{
	progress = (float)newProgress;
}

// Will be called on iPhone OS
- (void)setProgress:(float)newProgress;
{
	progress = newProgress;
}

#if TARGET_OS_IPHONE
- (void)testReachability
{
#if REACHABILITY_20_API
	NSLog(@"Using Reachability 2.0 API");
#else
	NSLog(@"Using Reachability 1.5 API");
#endif
	if ([ASIHTTPRequest isNetworkReachableViaWWAN]) {
		NSLog(@"Connected via WWAN");
	} else {
		NSLog(@"Not connected via WWAN");
	}
}
#endif

- (void)testAutomaticRetry
{
	ASIHTTPRequest *request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com"]];
	[request setTimeOutSeconds:0.001];
	[request setNumberOfTimesToRetryOnTimeout:5];
	[request startSynchronous];
	GHAssertNotNil([request error],@"Request failed to timeout, cannot proceed with test");
	BOOL success = ([request retryCount] == 5);
	GHAssertTrue(success,@"Request failed to retry on timeout");
}

- (void)testCopy
{
	NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
	
	ASIHTTPRequest *request = [ASIHTTPRequest requestWithURL:[NSURL URLWithString:@"http://allseeing-i.com"]];
	ASIHTTPRequest *request2 = [request copy];
	
	[pool release];
	
	GHAssertNotNil(request2,@"Failed to create a copy");
	BOOL success = ([request2 retainCount] == 1);
	GHAssertTrue(success,@"Failed to create a retained copy");
	
	[request2 release];
}

@end