/**
 * 
 */
package viewer;

import java.awt.*;
import java.awt.event.*;

import javax.media.opengl.GL;
import javax.media.opengl.GLCanvas;
import java.net.*;
import java.io.*;
import java.util.*;
// import java.awt.image.*;

import model.*;
import server.*;

/**
 * The window and general information for the viewer.
 * 
 * @author Henry Gross
 * @author Ed Williams
 */
public class Viewer {

	private static String path = null;

	private static String dataPath = null;

	public static String getPath() {
		if (path == null)
			try {
				path = URLDecoder.decode(Viewer.class.getProtectionDomain()
						.getCodeSource().getLocation().getPath(), "UTF-8");
			} catch (UnsupportedEncodingException e) {
			}
		return path;
	}

	public static String getDataPath() {
		if (dataPath == null)
			dataPath = getPath() + "basece/";
		return dataPath;
	}

	/**
	 * We want a way of closing the window when you click close
	 * 
	 * @author Henry
	 * 
	 */
	class ProgramTerminator implements WindowListener {
		public void windowClosing(WindowEvent event) {
			System.exit(0);
		}

		public void windowDeiconified(WindowEvent event) {
		}

		public void windowDeactivated(WindowEvent event) {
		}

		public void windowActivated(WindowEvent event) {
		}

		public void windowClosed(WindowEvent event) {
		}

		public void windowOpened(WindowEvent event) {
		}

		public void windowIconified(WindowEvent event) {
		}
	}

	private Frame window;

	private Panel view;

	private GLViewer glViewer;

	private InteractionListener interactionListener;

	private Interaction interaction;

	private Console console;

	private Model map;

	private ViewerClient client;

	private Options options;

	private HashMap<InetAddress, Double[]> people;

	private int x;

	private int y;

	private boolean fullscreen;

	/**
	 * Create a new viewer and load everything up
	 */
	public Viewer(boolean fullscreen, boolean startServer) {
		glViewer = new GLViewer(this);
		console = new Console();
		client = new ViewerClient();
		options = new Options(this);

		interaction = new Interaction(this, glViewer, console, options, client);
		interactionListener = new InteractionListener(interaction);
		interaction.attachInteractionListener(interactionListener);
		console.attachInteraction(interaction);
		options.attachInteraction(interaction);
		options.attachInteractionListener(interactionListener);
		client.attachInteraction(interaction);

		people = new HashMap<InetAddress, Double[]>();

		map = new Model();

		glViewer.setMap(map);
		glViewer.startGL();

		x = 640;
		y = 480;
		openWindow(fullscreen);

		if (startServer) {
			new CEDS();
			interaction.execute("/connect 127.0.0.1");
		}

		interaction.executeFile(Viewer.getDataPath() + "carlengine.cfg");
		interaction.executeFile(Viewer.getDataPath() + "autoexec");
		if (!window.isVisible())
			loadMap("blank/");
	}

	/**
	 * @param args
	 */
	public static void main(String[] args) {
		boolean fullscreen = true;
		boolean server = false;

		for (int i = 0; i < args.length; i++) {
			if (args[i].equals("-window"))
				fullscreen = false;
			if (args[i].equals("-server"))
				server = true;
		}

		new Viewer(fullscreen, server);
	}

	/**
	 * Create and open the prgram frame
	 */
	private void openWindow(boolean fullscreen) {
		window = new Frame("Viewer");
		window.setSize(x, y);

		view = new Panel(new BorderLayout());
		window.add(view);

		if (fullscreen)
			setFullscreen(true);

		window.addWindowListener(new ProgramTerminator());
		GLCanvas glCanvas = glViewer.getCanvas();

		view.add(glCanvas, BorderLayout.CENTER);
		view.add(console.getConsole(), BorderLayout.NORTH);
		view.add(options.getOptions(), BorderLayout.SOUTH);
		console.setVisible(false);
		options.setVisible(false);

		glCanvas.addKeyListener(interactionListener);
		glCanvas.addMouseListener(interactionListener);
		glCanvas.addMouseMotionListener(interactionListener);
		glCanvas.addMouseWheelListener(interactionListener);

		/*
		 * //make transparent image and set cursor to it int[] pixels = new
		 * int[16 * 16]; Image image = Toolkit.getDefaultToolkit().createImage(
		 * new MemoryImageSource(16, 16, pixels, 0, 16)); Cursor
		 * transparentCursor = Toolkit.getDefaultToolkit().createCustomCursor
		 * (image, new Point(0, 0), "invisiblecursor");
		 * window.setCursor(transparentCursor);
		 */

		// System.in might not work, but System.out does
		System.setOut(console.getOut());
		// System.setIn(console.getIn());
	}

	/**
	 * if something changes, his is the easiest way to refresh everything to
	 * make the changes be applied
	 */
	public void refreshWindow() {
		window.setVisible(true);
	}

	/**
	 * loads the map that the program will use
	 * 
	 * @param filename
	 *            the name of the file to load
	 */
	public void loadMap(String filename) {
		// TODO talk about this?
		// TODO: Make spaces in filenames okay! Aaaa!
		map
				.readModel(map, new File(Viewer.getDataPath() + "model/"
						+ filename));
		refreshWindow();
	}

	/**
	 * Set a specific model for the GLViewer, instead of loading a map
	 * 
	 * @param m
	 *            the model instance to use
	 */
	public void forceModel(Model m) {
		glViewer.setMap(m);
		glViewer.draw();
		// System.err.println("set map to " + map.getAlias());
		map = m;
	}

	public void forceModelSameGL(Model m) {
		forceModelSameGL(m, false);
	}

	public void forceModelSameGL(Model m, boolean forceRender) {
		glViewer.setMapSameGL(m);
		glViewer.setForceRenderNext(forceRender);
		glViewer.draw();
		// System.err.println("set map to " + map.getAlias());
		map = m;
	}

	/**
	 * Set the size of the window
	 * 
	 * @param x
	 *            the x dimension
	 * @param y
	 *            the y dimension
	 */
	public void setSize(int x, int y) {
		this.x = x;
		this.y = y;
		window.setSize(this.x, this.y);
		refreshWindow();
	}

	/**
	 * Update the position of the people
	 * 
	 * @param key
	 *            the key for the person to update
	 * @param x
	 *            the x coordinate
	 * @param y
	 *            the y coordinate
	 * @param z
	 *            the z coordinate
	 */
	public void updatePeople(String key, Double x, Double y, Double z) {
		try {
			InetAddress addrKey = InetAddress.getByName(key);
			if (!addrKey.toString().substring(1).equals(client.getAddress())) {
				if (people.containsKey(addrKey))
					people.remove(addrKey);
				Double[] value = new Double[] { x, y, z };
				people.put(addrKey, value);
			}
		} catch (UnknownHostException e) {
		}
	}

	/**
	 * return and array of the position of the people on the map
	 * 
	 * @return array of the position of the people on the map
	 */
	public ArrayList<Double[]> getPeople() {
		return new ArrayList<Double[]>(people.values());
	}

	/**
	 * Set whether the window is fullscreen is not
	 * 
	 * @param b
	 *            true to make the window fullscreen
	 */
	public void setFullscreen(boolean b) {
		window.dispose();
		if (b) {
			Toolkit tk = Toolkit.getDefaultToolkit();
			Dimension screenDimension = tk.getScreenSize();
			window.setUndecorated(true);
			window.setAlwaysOnTop(true);
			window.setSize((int) screenDimension.getWidth(),
					(int) screenDimension.getHeight());
		} else {
			window.setUndecorated(false);
			window.setAlwaysOnTop(false);
			window.setSize(x, y);
		}
		window.setVisible(true);
		fullscreen = b;
	}

	/**
	 * return the fullscreen state of the window
	 * 
	 * @return true if the window is fullscreen
	 */
	public boolean getFullscreen() {
		return fullscreen;
	}

	/**
	 * Return the size of the screem
	 * 
	 * @return the size of the screen
	 */
	public Rectangle getSize() {
		return new Rectangle(x, y);
	}

	/**
	 * Accesses the GL object belonging to the GLViewer. Specifically, used by
	 * The Preditor to save a new model.
	 * 
	 * @return GL belonging to the GLViewer
	 */
	public GL getGL() {
		return glViewer.getGL();
	}

	/**
	 * Accesses the GL viewer
	 * 
	 * @return the glViewer instance
	 */
	public GLViewer getGLViewer() {
		return glViewer;
	}

}
