/**
 * 
 */
package viewer;

import java.awt.*;
import java.awt.event.*;
import java.io.*;

import javax.swing.*;
import javax.swing.event.*;

/**
 * Make it easy to change some options other than just by using the command line
 * 
 * @author Henry
 * 
 */
public class Options {
	private JTabbedPane optionPane;

	private GridBagLayout layoutGeneral;

	private GridBagLayout layoutControls;

	private GridBagLayout layoutVideo;

	private OptionListener optionListener;

	private Interaction interaction;

	private InteractionListener interactionListener;

	private Viewer viewer;

	private JPanel general;

	private JPanel quitPanel;

	private JLabel quitLabel;

	private JButton quitButton;

	private JPanel controls;

	private JPanel sensitivityPanel;

	private JLabel sensitivityLabel;

	private JSlider sensitivitySlider;

	private JPanel forwardPanel;

	private JLabel forwardLabel;

	private JTextField forwardTextField;

	private JPanel backwardPanel;

	private JLabel backwardLabel;

	private JTextField backwardTextField;

	private JPanel rightPanel;

	private JLabel rightLabel;

	private JTextField rightTextField;

	private JPanel leftPanel;

	private JLabel leftLabel;

	private JTextField leftTextField;

	private JPanel lookUpPanel;

	private JLabel lookUpLabel;

	private JTextField lookUpTextField;;

	private JPanel lookDownPanel;

	private JLabel lookDownLabel;

	private JTextField lookDownTextField;

	private JPanel lookRightPanel;

	private JLabel lookRightLabel;

	private JTextField lookRightTextField;;

	private JPanel lookLeftPanel;

	private JLabel lookLeftLabel;

	private JTextField lookLeftTextField;

	private JPanel video;

	private JPanel resolutionPanel;

	private JLabel resolutionLabel;

	private JComboBox resolutionBox;

	private JPanel fullscreenPanel;

	private JLabel fullscreenLabel;

	private JButton fullscreenButton;

	private JPanel brightnessPanel;

	private JLabel brightnessLabel;

	private JSlider brightnessSlider;

	private JPanel texturePanel;

	private JLabel textureLabel;

	private JButton textureButton;

	/**
	 * Action Listener for all of the Options components
	 * 
	 * @author Henry
	 * 
	 */
	private class OptionListener implements ActionListener, ChangeListener,
			KeyListener {
		private Options options;

		Interaction interaction;

		public OptionListener(Options options) {
			this.options = options;
		}

		/**
		 * We need an interaction in order to do something so let's supply one
		 * 
		 * @param interaction
		 *            the interaction instance to use
		 */
		public void attachInteraction(Interaction interaction) {
			this.interaction = interaction;
		}

		public void actionPerformed(ActionEvent e) {
			if (e.getSource() == resolutionBox) {
				ResolutionItem res = (ResolutionItem) ((JComboBox) e
						.getSource()).getSelectedItem();
				interaction.execute("/resolution " + res.getX() + " "
						+ res.getY());
			} else if (e.getSource() == quitButton)
				interaction.execute("/quit");
			else if (e.getSource() == fullscreenButton) {
				if (fullscreenButton.getText().equals("On")) {
					interaction.execute("/fullscreen false");
					fullscreenButton.setText("Off");
				} else if (fullscreenButton.getText().equals("Off")) {
					interaction.execute("/fullscreen true");
					fullscreenButton.setText("On");
				}
			} else if (e.getSource() == textureButton)
				if (textureButton.getText().equals("On")) {
					interaction.execute("/textures false");
					textureButton.setText("Off");
				} else if (textureButton.getText().equals("Off")) {
					interaction.execute("/textures true");
					textureButton.setText("On");
				}
			options.saveOptions();
		}

		public void stateChanged(ChangeEvent e) {
			if (e.getSource() == brightnessSlider)
				interaction.execute("/brightness "
						+ ((JSlider) e.getSource()).getValue());
			else if (e.getSource() == sensitivitySlider)
				interaction.execute("/sensitivity "
						+ ((JSlider) e.getSource()).getValue());
			options.saveOptions();
		}

		public void keyPressed(KeyEvent e) {
			int keyCode = e.getKeyCode();
			String keyText = KeyEvent.getKeyText(keyCode);
			if (keyText == "Escape") {
				interaction.toggleOptions();
				e.consume();
			}

			if (e.getSource() == forwardTextField) {
				interaction.execute("/bind " + keyText + " +forward");
				forwardTextField.setText(keyText);
				e.consume();
			} else if (e.getSource() == backwardTextField) {
				interaction.execute("/bind " + keyText + " +back");
				backwardTextField.setText(keyText);
				e.consume();
			} else if (e.getSource() == rightTextField) {
				interaction.execute("/bind " + keyText + " +moveright");
				rightTextField.setText(keyText);
				e.consume();
			} else if (e.getSource() == leftTextField) {
				interaction.execute("/bind " + keyText + " +moveleft");
				leftTextField.setText(keyText);
				e.consume();
			} else if (e.getSource() == lookUpTextField) {
				interaction.execute("/bind " + keyText + " +lookup");
				lookUpTextField.setText(keyText);
				e.consume();
			} else if (e.getSource() == lookDownTextField) {
				interaction.execute("/bind " + keyText + " +lookdown");
				lookDownTextField.setText(keyText);
				e.consume();
			} else if (e.getSource() == lookRightTextField) {
				interaction.execute("/bind " + keyText + " +right");
				lookRightTextField.setText(keyText);
				e.consume();
			} else if (e.getSource() == lookLeftTextField) {
				interaction.execute("/bind " + keyText + " +left");
				lookLeftTextField.setText(keyText);
				e.consume();
			}
			refreshStates();
			options.saveOptions();
		}

		public void keyReleased(KeyEvent arg0) {
		}

		public void keyTyped(KeyEvent e) {
			e.consume();
		}
	}

	/**
	 * A class representing a screen resolution
	 * 
	 * @author Henry
	 * 
	 */
	private class ResolutionItem {
		private int x;

		private int y;

		public ResolutionItem(int x, int y) {
			this.x = x;
			this.y = y;
		}

		@Override
		public String toString() {
			return x + "X" + y;
		}

		/**
		 * get the x dimension of the resolution
		 * 
		 * @return the width of the resolution
		 */
		public int getX() {
			return x;
		}

		/**
		 * get the y dimension of the resolution
		 * 
		 * @return the width of the resolution
		 */
		public int getY() {
			return y;
		}
	}

	public Options(Viewer viewer) {
		this.viewer = viewer;

		layoutGeneral = new GridBagLayout();
		layoutControls = new GridBagLayout();
		layoutVideo = new GridBagLayout();

		GridBagConstraints c = new GridBagConstraints();
		c.fill = GridBagConstraints.HORIZONTAL;
		c.weightx = 0.5;
		c.weighty = 0.5;

		optionListener = new OptionListener(this);
		optionPane = new JTabbedPane();
		optionPane.addKeyListener(optionListener);

		general = new JPanel(layoutGeneral);
		general.setPreferredSize(new Dimension(200, 50));

		quitPanel = new JPanel();
		quitLabel = new JLabel("Quit");
		quitButton = new JButton("Quit");
		quitButton.addActionListener(optionListener);
		quitPanel.add(quitLabel);
		quitPanel.add(quitButton);

		general.add(quitPanel);

		controls = new JPanel(layoutControls);
		controls.setPreferredSize(new Dimension(400, 200));

		sensitivityPanel = new JPanel();
		sensitivityLabel = new JLabel("Movement Sensitivity");
		sensitivitySlider = new JSlider(1, 100, 1);
		sensitivitySlider.addChangeListener(optionListener);
		sensitivityPanel.add(sensitivityLabel);
		sensitivityPanel.add(sensitivitySlider);

		forwardPanel = new JPanel();
		forwardLabel = new JLabel("Forward");
		forwardTextField = new JTextField("A");
		forwardTextField.setColumns(10);
		forwardTextField.addKeyListener(optionListener);
		forwardPanel.add(forwardLabel);
		forwardPanel.add(forwardTextField);

		backwardPanel = new JPanel();
		backwardLabel = new JLabel("Backward");
		backwardTextField = new JTextField("S");
		backwardTextField.setColumns(10);
		backwardTextField.addKeyListener(optionListener);
		backwardPanel.add(backwardLabel);
		backwardPanel.add(backwardTextField);

		rightPanel = new JPanel();
		rightLabel = new JLabel("Right");
		rightTextField = new JTextField("D");
		rightTextField.setColumns(10);
		rightTextField.addKeyListener(optionListener);
		rightPanel.add(rightLabel);
		rightPanel.add(rightTextField);

		leftPanel = new JPanel();
		leftLabel = new JLabel("Left");
		leftTextField = new JTextField("W");
		leftTextField.setColumns(10);
		leftTextField.addKeyListener(optionListener);
		leftPanel.add(leftLabel);
		leftPanel.add(leftTextField);

		lookUpPanel = new JPanel();
		lookUpLabel = new JLabel("Look Up");
		lookUpTextField = new JTextField("Up");
		lookUpTextField.setColumns(10);
		lookUpTextField.addKeyListener(optionListener);
		lookUpPanel.add(lookUpLabel);
		lookUpPanel.add(lookUpTextField);

		lookDownPanel = new JPanel();
		lookDownLabel = new JLabel("Look Down");
		lookDownTextField = new JTextField("Down");
		lookDownTextField.setColumns(10);
		lookDownTextField.addKeyListener(optionListener);
		lookDownPanel.add(lookDownLabel);
		lookDownPanel.add(lookDownTextField);

		lookRightPanel = new JPanel();
		lookRightLabel = new JLabel("Look Right");
		lookRightTextField = new JTextField("Right");
		lookRightTextField.setColumns(10);
		lookRightTextField.addKeyListener(optionListener);
		lookRightPanel.add(lookRightLabel);
		lookRightPanel.add(lookRightTextField);

		lookLeftPanel = new JPanel();
		lookLeftLabel = new JLabel("Look Left");
		lookLeftTextField = new JTextField("Left");
		lookLeftTextField.setColumns(10);
		lookLeftTextField.addKeyListener(optionListener);
		lookLeftPanel.add(lookLeftLabel);
		lookLeftPanel.add(lookLeftTextField);

		c.gridx = 0;
		c.gridy = 0;
		c.gridwidth = 2;
		controls.add(sensitivityPanel, c);
		c.gridx = 0;
		c.gridy = 1;
		c.gridwidth = 1;
		controls.add(forwardPanel, c);
		c.gridx = 1;
		c.gridy = 1;
		controls.add(lookUpPanel, c);
		c.gridx = 0;
		c.gridy = 2;
		controls.add(backwardPanel, c);
		c.gridx = 1;
		c.gridy = 2;
		controls.add(lookDownPanel, c);
		c.gridx = 0;
		c.gridy = 3;
		controls.add(rightPanel, c);
		c.gridx = 1;
		c.gridy = 3;
		controls.add(lookRightPanel, c);
		c.gridx = 0;
		c.gridy = 4;
		controls.add(leftPanel, c);
		c.gridx = 1;
		c.gridy = 4;
		controls.add(lookLeftPanel, c);

		video = new JPanel(layoutVideo);
		video.setPreferredSize(new Dimension(400, 100));

		resolutionPanel = new JPanel();
		resolutionLabel = new JLabel("Resolution");
		resolutionBox = new JComboBox();
		resolutionBox.addItem(new ResolutionItem(640, 480));
		resolutionBox.addItem(new ResolutionItem(800, 500));
		resolutionBox.addItem(new ResolutionItem(800, 600));
		resolutionBox.addItem(new ResolutionItem(1024, 600));
		resolutionBox.addItem(new ResolutionItem(1024, 768));
		resolutionBox.addItem(new ResolutionItem(1280, 800));
		resolutionBox.addItem(new ResolutionItem(1280, 1024));
		resolutionBox.addActionListener(optionListener);
		resolutionPanel.add(resolutionLabel);
		resolutionPanel.add(resolutionBox);

		fullscreenPanel = new JPanel();
		fullscreenLabel = new JLabel("Fullscreen");
		fullscreenButton = new JButton("Off");
		fullscreenButton.setPreferredSize(new Dimension(51, 26));
		fullscreenButton.addActionListener(optionListener);
		fullscreenPanel.add(fullscreenLabel);
		fullscreenPanel.add(fullscreenButton);

		brightnessPanel = new JPanel();
		brightnessLabel = new JLabel("Brightness");
		brightnessSlider = new JSlider(0, 100, 100);
		brightnessSlider.addChangeListener(optionListener);
		brightnessPanel.add(brightnessLabel);
		brightnessPanel.add(brightnessSlider);

		texturePanel = new JPanel();
		textureLabel = new JLabel("Textures");
		textureButton = new JButton("Off");
		textureButton.setPreferredSize(new Dimension(51, 26));
		textureButton.addActionListener(optionListener);
		texturePanel.add(textureLabel);
		texturePanel.add(textureButton);

		c.gridx = 0;
		c.gridy = 0;
		video.add(resolutionPanel, c);
		c.gridx = 1;
		c.gridy = 0;
		video.add(fullscreenPanel, c);
		c.gridx = 0;
		c.gridy = 1;
		video.add(brightnessPanel, c);
		c.gridx = 1;
		c.gridy = 1;
		video.add(texturePanel, c);

		optionPane.addTab("General", general);
		optionPane.addTab("Controls", controls);
		optionPane.addTab("Video", video);
	}

	/**
	 * give the options instance the interaction instance to use
	 * 
	 * @param interaction
	 *            the interaction instance to use
	 */
	public void attachInteraction(Interaction interaction) {
		this.interaction = interaction;
		optionListener.attachInteraction(this.interaction);
	}

	/**
	 * give the options instance the interactionlistener instance to use
	 * 
	 * @param interactionListener
	 *            ine interactionListener instance to use
	 */
	public void attachInteractionListener(
			InteractionListener interactionListener) {
		this.interactionListener = interactionListener;
	}

	/**
	 * get the optionPane to display
	 * 
	 * @return the optionPane
	 */
	public JTabbedPane getOptions() {
		return optionPane;
	}

	/**
	 * Refresh the states of the various options to make sure they are correct
	 */
	public void refreshStates() {
		sensitivitySlider.setValue((int) interaction.getSensitivity());
		if (viewer.getFullscreen())
			fullscreenButton.setText("On");
		else if (!viewer.getFullscreen())
			fullscreenButton.setText("Off");
		if (viewer.getGLViewer().getTextureState())
			textureButton.setText("On");
		else if (!viewer.getGLViewer().getTextureState())
			textureButton.setText("Off");
		forwardTextField.setText(interactionListener
				.getKey(InteractionListener.CFG_MOVE_FORWARD));
		backwardTextField.setText(interactionListener
				.getKey(InteractionListener.CFG_MOVE_BACKWARD));
		rightTextField.setText(interactionListener
				.getKey(InteractionListener.CFG_MOVE_RIGHT));
		leftTextField.setText(interactionListener
				.getKey(InteractionListener.CFG_MOVE_LEFT));
		lookUpTextField.setText(interactionListener
				.getKey(InteractionListener.CFG_LOOK_UP));
		lookDownTextField.setText(interactionListener
				.getKey(InteractionListener.CFG_LOOK_DOWN));
		lookRightTextField.setText(interactionListener
				.getKey(InteractionListener.CFG_LOOK_RIGHT));
		lookLeftTextField.setText(interactionListener
				.getKey(InteractionListener.CFG_LOOK_LEFT));
	}

	/**
	 * Show or hide the options
	 * 
	 * @param b
	 *            true to display the options
	 */
	public void setVisible(boolean b) {
		optionPane.setVisible(b);
	}

	/**
	 * Get the visibility of the options
	 * 
	 * @return true if the options is visible
	 */
	public boolean isVisible() {
		return optionPane.isVisible();
	}

	/**
	 * Request that the options receive focus
	 */
	public void requestFocus() {
		optionPane.requestFocus();
	}

	/**
	 * Save the settings to the config file
	 * 
	 */
	public void saveOptions() {
		try {
			FileWriter optionsFile = new FileWriter(Viewer.getDataPath()
					+ "carlengine.cfg");
			optionsFile.write("bind "
					+ interactionListener
							.getKey(InteractionListener.CFG_MOVE_FORWARD)
					+ " +forward" + "\r\n");
			optionsFile.write("bind "
					+ interactionListener
							.getKey(InteractionListener.CFG_MOVE_BACKWARD)
					+ " +back" + "\r\n");
			optionsFile.write("bind "
					+ interactionListener
							.getKey(InteractionListener.CFG_MOVE_RIGHT)
					+ " +moveright" + "\r\n");
			optionsFile.write("bind "
					+ interactionListener
							.getKey(InteractionListener.CFG_MOVE_LEFT)
					+ " +moveleft" + "\r\n");
			optionsFile.write("bind "
					+ interactionListener
							.getKey(InteractionListener.CFG_LOOK_UP)
					+ " +lookup" + "\r\n");
			optionsFile.write("bind "
					+ interactionListener
							.getKey(InteractionListener.CFG_LOOK_DOWN)
					+ " +lookdown" + "\r\n");
			optionsFile.write("bind "
					+ interactionListener
							.getKey(InteractionListener.CFG_LOOK_RIGHT)
					+ " +right" + "\r\n");
			optionsFile.write("bind "
					+ interactionListener
							.getKey(InteractionListener.CFG_LOOK_LEFT)
					+ " +left" + "\r\n");
			optionsFile.close();
		} catch (IOException e) {
		}
	}
}
