/**
 * 
 */
package viewer;

import java.awt.event.*;
import java.awt.Robot;
import java.util.*;

/**
 * Action listener for interaction.
 * 
 * @author Henry Gross
 * @author Ed Williams
 */
public class InteractionListener implements KeyListener, MouseListener,
		MouseMotionListener, MouseWheelListener {

	private Interaction interaction;

	private HashMap<String, Integer> cfgHashMap;

	private HashMap<String, String> cmdHashMap;

	private String[] cfgArr;

	public static final int CFG_COMMAND = 0;

	public static final int CFG_MOVE_FORWARD = 2;

	public static final int CFG_MOVE_BACKWARD = 3;

	public static final int CFG_MOVE_RIGHT = 4;

	public static final int CFG_MOVE_LEFT = 5;

	public static final int CFG_LOOK_UP = 6;

	public static final int CFG_LOOK_DOWN = 7;

	public static final int CFG_LOOK_RIGHT = 8;

	public static final int CFG_LOOK_LEFT = 9;

	public static final int CFG_MOUSE = 10;

	private int x;

	private int y;

	private int x_threshold = 0;

	private int y_threshold = 0;

	private boolean ROBOT_MOUSE = true;

	private Robot robot;

	private int xCenter;

	private int yCenter;

	private boolean mouseMove = false;

	public static final int SENSITIVITY_LEVEL = 10;

	/**
	 * create a new instance and set up some default keybindings
	 */
	public InteractionListener(Interaction interaction) {
		this.interaction = interaction;
		cfgHashMap = new HashMap<String, Integer>();
		cmdHashMap = new HashMap<String, String>();
		cfgArr = new String[CFG_MOUSE + 1];

		bindKey("W", CFG_MOVE_FORWARD);
		bindKey("S", CFG_MOVE_BACKWARD);
		bindKey("D", CFG_MOVE_RIGHT);
		bindKey("A", CFG_MOVE_LEFT);

		bindKey("Up", CFG_LOOK_UP);
		bindKey("Down", CFG_LOOK_DOWN);
		bindKey("Right", CFG_LOOK_RIGHT);
		bindKey("Left", CFG_LOOK_LEFT);

		bindKey("H", "/spawn");

		bindKey("M", CFG_MOUSE);

		xCenter = interaction.getWidth() / 2;
		yCenter = interaction.getHeight() / 2;

		try {
			robot = new Robot();
		} catch (Exception e) {
			System.out.println("error with mouse");
		}
	}

	/**
	 * bind a key to something common enogh we have it as a final int
	 * 
	 * @param keyText
	 *            the key to be bound
	 * @param command
	 *            the command it should perform
	 */
	public void bindKey(String keyText, int command) {
		int delCmd = -1;
		if (cfgHashMap.containsKey(keyText)) {
			delCmd = cfgHashMap.get(keyText);
			cfgHashMap.remove(keyText);
		}
		cfgHashMap.put(keyText, command);
		cfgArr[command] = keyText;
		if (delCmd > 0 && delCmd != command)
			cfgArr[delCmd] = "";
	}

	/**
	 * set a key to a general command to be parsed by Interaction
	 * 
	 * @param keyText
	 *            the key to be bound
	 * @param cmd
	 *            the command to be performed
	 */
	public void bindKey(String keyText, String cmd) {
		bindKey(keyText, CFG_COMMAND);
		if (cmdHashMap.containsKey(keyText))
			cmdHashMap.remove(keyText);
		cmdHashMap.put(keyText, cmd);
	}

	/**
	 * bind a key, but it might involve a bit of parsing first
	 * 
	 * @param bindCmd
	 *            the array list to be parsed to figure out the key binding
	 */
	public void bind(ArrayList<String> bindCmd) {
		int cmd = CFG_COMMAND;
		String key = "";
		String cmdString = "";

		for (int i = 2; i < bindCmd.size(); i++)
			cmdString += bindCmd.get(i) + " ";
		key = bindCmd.get(1);
		if (bindCmd.get(2).equals("+forward"))
			cmd = CFG_MOVE_FORWARD;
		if (bindCmd.get(2).equals("+back"))
			cmd = CFG_MOVE_BACKWARD;
		if (bindCmd.get(2).equals("+moveright"))
			cmd = CFG_MOVE_RIGHT;
		if (bindCmd.get(2).equals("+moveleft"))
			cmd = CFG_MOVE_LEFT;
		if (bindCmd.get(2).equals("+lookdown"))
			cmd = CFG_LOOK_DOWN;
		if (bindCmd.get(2).equals("+lookup"))
			cmd = CFG_LOOK_UP;
		if (bindCmd.get(2).equals("+right"))
			cmd = CFG_LOOK_RIGHT;
		if (bindCmd.get(2).equals("+left"))
			cmd = CFG_LOOK_LEFT;

		if (cmd == CFG_COMMAND)
			bindKey(key, cmdString);
		else
			bindKey(key, cmd);
	}

	/**
	 * act accordingly to whatever key was pressed
	 */
	public void keyPressed(KeyEvent e) {
		int keyCode = e.getKeyCode();
		String keyText = KeyEvent.getKeyText(keyCode);

		if (keyText == "Back Quote")
			interaction.toggleConsole();
		else if (keyText == "Escape")
			interaction.toggleOptions();
		else if (keyText == "F12")
			interaction.takeScreenshot();

		if (cfgHashMap.containsKey(keyText)) {
			int method = Integer.valueOf(cfgHashMap.get(keyText));
			switch (method) {
			case CFG_COMMAND:
				interaction.execute(cmdHashMap.get(keyText));
				break;
			case CFG_MOVE_FORWARD:
				interaction.move(Interaction.MOVE_FORWARD, true);
				break;
			case CFG_MOVE_BACKWARD:
				interaction.move(Interaction.MOVE_BACKWARD, true);
				break;
			case CFG_MOVE_RIGHT:
				interaction.move(Interaction.MOVE_RIGHT, true);
				break;
			case CFG_MOVE_LEFT:
				interaction.move(Interaction.MOVE_LEFT, true);
				break;
			case CFG_LOOK_UP:
				interaction.move(Interaction.LOOK_UP, true);
				break;
			case CFG_LOOK_DOWN:
				interaction.move(Interaction.LOOK_DOWN, true);
				break;
			case CFG_LOOK_RIGHT:
				interaction.move(Interaction.LOOK_RIGHT, true);
				break;
			case CFG_LOOK_LEFT:
				interaction.move(Interaction.LOOK_LEFT, true);
				break;
			case CFG_MOUSE:
				if (mouseMove)
					mouseMove = false;
				else {
					mouseMove = true;
					xCenter = interaction.getWidth() / 2;
					yCenter = interaction.getHeight() / 2;
				}
				break;

			}
		}
	}

	/**
	 * we want to know when a key was released so we can stop performing
	 * whatever it was performing
	 */
	public void keyReleased(KeyEvent e) {
		int keyCode = e.getKeyCode();
		String keyText = KeyEvent.getKeyText(keyCode);
		if (cfgHashMap.containsKey(keyText)) {
			int method = Integer.valueOf(cfgHashMap.get(keyText));
			switch (method) {
			case CFG_MOVE_FORWARD:
				interaction.move(Interaction.MOVE_FORWARD, false);
				break;
			case CFG_MOVE_BACKWARD:
				interaction.move(Interaction.MOVE_BACKWARD, false);
				break;
			case CFG_MOVE_RIGHT:
				interaction.move(Interaction.MOVE_RIGHT, false);
				break;
			case CFG_MOVE_LEFT:
				interaction.move(Interaction.MOVE_LEFT, false);
				break;
			case CFG_LOOK_UP:
				interaction.move(Interaction.LOOK_UP, false);
				break;
			case CFG_LOOK_DOWN:
				interaction.move(Interaction.LOOK_DOWN, false);
				break;
			case CFG_LOOK_RIGHT:
				interaction.move(Interaction.LOOK_RIGHT, false);
				break;
			case CFG_LOOK_LEFT:
				interaction.move(Interaction.LOOK_LEFT, false);
				break;
			}
		}
	}

	public void keyTyped(KeyEvent e) {
	}

	public void mouseClicked(MouseEvent e) {
		/*
		 * xCenter = interaction.getWidth() / 2; yCenter =
		 * interaction.getHeight() / 2; System.out.println(xCenter + " "
		 * +yCenter);
		 */
	}

	public void mouseEntered(MouseEvent e) {

	}

	public void mouseExited(MouseEvent e) {
	}

	public void mousePressed(MouseEvent e) {
		y = e.getY();
		x = e.getX();

		// System.out.println(xCenter + " " +yCenter);
	}

	public void mouseReleased(MouseEvent e) {
	}

	public void mouseDragged(MouseEvent e) {

		int newx = e.getX();
		int newy = e.getY();

		x_threshold = x_threshold + newx - x;
		y_threshold = y_threshold + newy - y;

		if (Math.abs(x_threshold) > SENSITIVITY_LEVEL) {
			if (x_threshold > 0) {
				interaction.move(Interaction.LOOK_RIGHT, true);
				interaction.move(Interaction.LOOK_RIGHT, false);
			} else {
				interaction.move(Interaction.LOOK_LEFT, true);
				interaction.move(Interaction.LOOK_LEFT, false);
			}
			x_threshold = 0;
		}

		if (Math.abs(y_threshold) > SENSITIVITY_LEVEL) {
			if (y_threshold > 0) {
				interaction.move(Interaction.LOOK_DOWN, true);
				interaction.move(Interaction.LOOK_DOWN, false);
			} else {
				interaction.move(Interaction.LOOK_UP, true);
				interaction.move(Interaction.LOOK_UP, false);
			}
			y_threshold = 0;
		}
		y = newy;
		x = newx;

	}

	public void mouseMoved(MouseEvent e) {
		if (mouseMove)
			if (ROBOT_MOUSE) {
				x = xCenter;
				y = yCenter;
				robot.mouseMove(x, y);
				ROBOT_MOUSE = false;

			} else {

				int newx = e.getX();
				int newy = e.getY();
				System.out.println(newx + " " + newy);
				x_threshold = x_threshold + newx - x;
				y_threshold = y_threshold + newy - y;

				if (Math.abs(x_threshold) > SENSITIVITY_LEVEL) {
					if (x_threshold > 0) {
						interaction.move(Interaction.LOOK_RIGHT, true);
						interaction.move(Interaction.LOOK_RIGHT, false);
					} else {
						interaction.move(Interaction.LOOK_LEFT, true);
						interaction.move(Interaction.LOOK_LEFT, false);
					}
					x_threshold = 0;
					ROBOT_MOUSE = true;

				}

				if (Math.abs(y_threshold) > SENSITIVITY_LEVEL) {
					if (y_threshold > 0) {
						interaction.move(Interaction.LOOK_DOWN, true);
						interaction.move(Interaction.LOOK_DOWN, false);
					} else {
						interaction.move(Interaction.LOOK_UP, true);
						interaction.move(Interaction.LOOK_UP, false);
					}
					y_threshold = 0;
					ROBOT_MOUSE = true;
				}
				y = newy;
				x = newx;

			}
	}

	public void mouseWheelMoved(MouseWheelEvent e) {
		if (e.getUnitsToScroll() > 0)
			interaction.decreaseSensitivity();
		else if (e.getUnitsToScroll() < 0)
			interaction.increaseSensitivity();
	}

	/**
	 * get the key which is bound to the given command
	 * 
	 * @param cmd
	 *            the command to find the key for
	 * @return the key bound to cmd
	 */
	public String getKey(int cmd) {
		return cfgArr[cmd];
	}

}
