/**
 * 
 */
package viewer;

import javax.media.opengl.*;

import java.util.*;
import java.io.*;
import server.*;

/**
 * Bridges the gap between input and the functions that need to be called to
 * interact with the viewer.
 * 
 * @author Henry Gross
 * @author Ed Williams
 */
public class Interaction {

	private Viewer viewer;

	private GLViewer glViewer;

	private Console console;

	private Options options;

	private Client client;

	private InteractionListener interactionListener;

	private Scanner cmdScan;

	private ArrayList<String> cmdArr;

	public static final int MOVE_FORWARD = 0;

	public static final int MOVE_BACKWARD = 1;

	public static final int MOVE_RIGHT = 2;

	public static final int MOVE_LEFT = 3;

	public static final int LOOK_UP = 4;

	public static final int LOOK_DOWN = 5;

	public static final int LOOK_RIGHT = 6;

	public static final int LOOK_LEFT = 7;

	private boolean freeMove;

	private boolean[] movement;

	private double sensitivity;

	/**
	 * create a new interaction instance and establish to links to everything it
	 * might interact with
	 */
	public Interaction(Viewer viewer, GLViewer glViewer, Console console,
			Options options, Client client) {
		this.viewer = viewer;
		this.glViewer = glViewer;
		this.console = console;
		this.options = options;
		this.client = client;

		movement = new boolean[8];
		for (int i = 0; i < movement.length; i++)
			movement[i] = false;
		freeMove = true;
		sensitivity = 100;
	}

	/**
	 * add an interaction listener to since we have to send some commands there
	 * 
	 * @param interactionListener
	 *            the interactionListeer to attach to
	 */
	public void attachInteractionListener(
			InteractionListener interactionListener) {
		this.interactionListener = interactionListener;
	}

	/**
	 * execute all commands in a file
	 * 
	 * @param filename
	 *            the name of the file to execute
	 */
	public void executeFile(String filename) {
		try {
			Scanner fileReader = new Scanner(new File(filename));
			while (fileReader.hasNextLine())
				execute("/" + fileReader.nextLine());
			fileReader.close();
		} catch (FileNotFoundException e) {
			System.err.println("file not found");
		}
	}

	/**
	 * parse commands the server send us mostly just send them to execute, but
	 * we might not want to execute all commands the server sends us
	 * 
	 * @param command
	 *            the command to execute
	 */
	public void serverExecute(String command) {
		cmdScan = new Scanner(command);
		cmdArr = new ArrayList<String>();
		while (cmdScan.hasNext())
			cmdArr.add(cmdScan.next());
		if (cmdArr.get(0).equals("_pos")) {
			viewer.updatePeople(cmdArr.get(1), Double
					.parseDouble(cmdArr.get(2)), Double.parseDouble(cmdArr
					.get(3)), Double.parseDouble(cmdArr.get(4)));
			glViewer.draw();
		} else
			execute(command);
	}

	/**
	 * parse and execute a command
	 * 
	 * @param command
	 *            the command to execute
	 */
	public void execute(String command) {
		if (command.length() == 0)
			return;
		if (command.charAt(0) != '/') {
			say(command);
			return;
		}

		cmdScan = new Scanner(command);
		cmdArr = new ArrayList<String>();
		while (cmdScan.hasNext())
			cmdArr.add(cmdScan.next());
		if (cmdArr.get(0).equals("/quit") && cmdArr.size() == 1)
			quit();
		else if (cmdArr.get(0).equals("/spawn") && cmdArr.size() == 1)
			spawn();
		else if (cmdArr.get(0).equals("/map") && cmdArr.size() == 2)
			viewer.loadMap(cmdArr.get(1));
		else if (cmdArr.get(0).equals("/2D") && cmdArr.size() == 1) {
			glViewer.set3D(false);
			glViewer.draw();
		} else if (cmdArr.get(0).equals("/3D") && cmdArr.size() == 1) {
			glViewer.set3D(true);
			glViewer.draw();
		} else if (cmdArr.get(0).equals("/bind") && cmdArr.size() >= 3)
			interactionListener.bind(cmdArr);
		else if (cmdArr.get(0).equals("/cg_fov") && cmdArr.size() == 2) {
			glViewer.setFov(Double.parseDouble(cmdArr.get(1)));
			glViewer.draw();
		} else if (cmdArr.get(0).equals("/freemove") && cmdArr.size() == 2) {
			setFreeMove(cmdArr.get(1));
			glViewer.draw();
		} else if (cmdArr.get(0).equals("/cam") && cmdArr.size() == 1) {
			double[] pos = glViewer.getCam();
			System.out.println(pos[0] + " " + pos[1] + " " + pos[2] + " "
					+ pos[3] + " " + pos[4]);
		} else if (cmdArr.get(0).equals("/cam") && cmdArr.size() == 6) {
			double[] cam = { Double.parseDouble(cmdArr.get(1)),
					Double.parseDouble(cmdArr.get(2)),
					Double.parseDouble(cmdArr.get(3)),
					Double.parseDouble(cmdArr.get(4)),
					Double.parseDouble(cmdArr.get(5)) };
			glViewer.setCam(cam);
			glViewer.draw();
		} else if (cmdArr.get(0).equals("/cam") && cmdArr.size() == 1) {
			double[] cam = glViewer.getCam();
			System.out.println("Current cam location: " + cam[0] + ", "
					+ cam[1] + ", " + cam[2]);
			System.out.println("Current cam direction: " + cam[3] + ", "
					+ cam[4]);
		} else if (cmdArr.get(0).equals("/sensitivity") && cmdArr.size() == 2)
			setSensitivity(Float.parseFloat(cmdArr.get(1)));
		else if (cmdArr.get(0).equals("/resolution") && cmdArr.size() == 3)
			viewer.setSize(Integer.parseInt(cmdArr.get(1)), Integer
					.parseInt(cmdArr.get(2)));
		else if (cmdArr.get(0).equals("/textures") && cmdArr.size() == 2)
			glViewer.showTextures(Boolean.parseBoolean(cmdArr.get(1)));
		else if (cmdArr.get(0).equals("/fullscreen") && cmdArr.size() == 2)
			viewer.setFullscreen(Boolean.parseBoolean(cmdArr.get(1)));
		else if (cmdArr.get(0).equals("/connect") && cmdArr.size() == 2)
			try {
				client.connect(cmdArr.get(1));
				client.start();
			} catch (IOException e) {
			}
		else if (cmdArr.get(0).equals("/disconnect") && cmdArr.size() == 1)
			try {
				client.disconnect();
			} catch (IOException e) {
			}
		else if (cmdArr.get(0).equals("/ping") && cmdArr.size() == 1)
			client.ping();
		else if (cmdArr.get(0).equals("/pizza") && cmdArr.size() == 1) {
		} else if (cmdArr.get(0).equals("/brightness") && cmdArr.size() == 2)
			glViewer.changeBrightness(Integer.parseInt(cmdArr.get(1)));
		else if (cmdArr.get(0).equals("/diffuse") && cmdArr.size() == 4) {
			glViewer.lights.lightDiffuse(Float.parseFloat(cmdArr.get(1)), Float
					.parseFloat(cmdArr.get(2)),
					Float.parseFloat(cmdArr.get(3)), 0);
			glViewer.updateLights(true);
		} else if (cmdArr.get(0).equals("/ambient") && cmdArr.size() == 4) {
			glViewer.lights.lightAmbient(Float.parseFloat(cmdArr.get(1)), Float
					.parseFloat(cmdArr.get(2)),
					Float.parseFloat(cmdArr.get(3)), 0);
			glViewer.updateLights(true);
		} else if (cmdArr.get(0).equals("/lights") && cmdArr.size() == 2) {
			if (cmdArr.get(1).equals("on"))
				glViewer.setLights(true);
			if (cmdArr.get(1).equals("off"))
				glViewer.setLights(false);
		} else
			System.out.println("command not recognized");
	}

	/**
	 * simply print something to stdout
	 * 
	 * @param text
	 */
	public void say(String text) {
		System.out.println(text);
	}

	/**
	 * move in a certain direction
	 * 
	 * @param dir
	 *            the direction to move
	 * @param state
	 *            true if we should move in that direction
	 */
	public void move(int dir, boolean state) {
		movement[dir] = state;
		move();
		glViewer.draw();
	}

	/**
	 * go back to the spawn (start) point
	 * 
	 */
	public void spawn() {
		double[] spawnPoint = new double[] { 0, 0, 10000, 90, -85 };
		glViewer.setCam(spawnPoint);
		glViewer.draw();
	}

	/**
	 * toggle whether to display the console or not
	 * 
	 */
	public void toggleConsole() {
		if (!options.isVisible()) {
			console.setVisible(!console.isVisible());
			viewer.refreshWindow();
			if (console.isVisible())
				console.requestFocus();
			else
				glViewer.requestFocus();
		}
	}

	/**
	 * toggle whether to display the options pane or not
	 * 
	 */
	public void toggleOptions() {
		console.setVisible(false);
		options.setVisible(!options.isVisible());
		if (options.isVisible())
			options.refreshStates();
		viewer.refreshWindow();
		if (options.isVisible())
			options.requestFocus();
		else
			glViewer.requestFocus();
	}

	/**
	 * get the height of the glCanvas
	 * 
	 * @return the height of the glCanvas
	 */
	public int getHeight() {
		GLCanvas glCanvas = glViewer.getCanvas();
		int h = glCanvas.getHeight();
		return h;
	}

	/**
	 * get the width of the glCanvas
	 * 
	 * @return the width of the glCanvas
	 */
	public int getWidth() {
		GLCanvas glCanvas = glViewer.getCanvas();
		int w = glCanvas.getWidth();
		return w;
	}

	/**
	 * actually move the camera
	 */
	private void move() {
		double[] cam = glViewer.getCam();

		double xyLookRightDeg;
		double xLook;
		double yLook;
		double zLook;
		double xLookRight;
		double yLookRight;

		int moveForwardScale;
		int moveRightScale;
		int lookUpScale;
		int lookRightScale;

		xyLookRightDeg = cam[3] - 90;
		if (xyLookRightDeg < 0)
			xyLookRightDeg += 360;

		xLook = Math.cos(Math.toRadians(cam[3]))
				* Math.cos(Math.toRadians(cam[4]));
		yLook = Math.sin(Math.toRadians(cam[3]))
				* Math.cos(Math.toRadians(cam[4]));
		zLook = Math.sin(Math.toRadians(cam[4]))
				* Math.cos(Math.toRadians(cam[4]));
		xLookRight = Math.cos(Math.toRadians(xyLookRightDeg))
				* Math.cos(Math.toRadians(cam[4]));
		yLookRight = Math.sin(Math.toRadians(xyLookRightDeg))
				* Math.cos(Math.toRadians(cam[4]));

		if (movement[MOVE_FORWARD] && !movement[MOVE_BACKWARD])
			moveForwardScale = 1;
		else if (movement[MOVE_BACKWARD] && !movement[MOVE_FORWARD])
			moveForwardScale = -1;
		else
			moveForwardScale = 0;

		if (movement[MOVE_RIGHT] && !movement[MOVE_LEFT])
			moveRightScale = 1;
		else if (movement[MOVE_LEFT] && !movement[MOVE_RIGHT])
			moveRightScale = -1;
		else
			moveRightScale = 0;

		if (movement[LOOK_UP] && !movement[LOOK_DOWN])
			lookUpScale = 1;
		else if (movement[LOOK_DOWN] && !movement[LOOK_UP])
			lookUpScale = -1;
		else
			lookUpScale = 0;

		if (movement[LOOK_RIGHT] && !movement[LOOK_LEFT])
			lookRightScale = 1;
		else if (movement[LOOK_LEFT] && !movement[LOOK_RIGHT])
			lookRightScale = -1;
		else
			lookRightScale = 0;

		double x = xLook * moveForwardScale + xLookRight * moveRightScale;
		double y = yLook * moveForwardScale + yLookRight * moveRightScale;
		double z = zLook * moveForwardScale;

		double length = Math.sqrt(Math.pow(x, 2) + Math.pow(y, 2)
				+ Math.pow(z, 2));
		if (length != 0) {
			x *= sensitivity / length;
			y *= sensitivity / length;
			z *= sensitivity / length;
		}

		cam[3] -= 5 * lookRightScale;
		cam[4] += 5 * lookUpScale;

		if (cam[3] < 0)
			cam[3] += 360;
		else if (cam[3] > 360)
			cam[3] -= 360;
		if (cam[4] > 85)
			cam[4] = 85;
		else if (cam[4] < -85)
			cam[4] = -85;

		cam[0] += x;
		cam[1] += y;
		cam[2] += z;

		if (!freeMove)
			cam[2] = 50;

		glViewer.setCam(cam);

		if (client.isConnected())
			try {
				client.sendString("/asv_pos " + client.getAddress() + " "
						+ cam[0] + " " + cam[1] + " " + cam[2]);
			} catch (IOException e) {
			}
	}

	/**
	 * Return the current sensitivity setting
	 * 
	 * @return the current sensitivity setting
	 */
	public double getSensitivity() {
		return sensitivity;
	}

	/**
	 * set the sensitivity to a specific amount
	 * 
	 * @param sens
	 *            the level to set the sensitivity at
	 */
	public void setSensitivity(double sens) {
		if (sens > 1000)
			sens = 1000;
		else if (sens < 1)
			sens = 1;
		sensitivity = sens;
	}

	/**
	 * Increase the sensitivity of movement
	 */
	public void increaseSensitivity() {
		sensitivity += 2;
		if (sensitivity > 1000)
			sensitivity = 1000;
	}

	/**
	 * Decrease the sensitivity of movement
	 * 
	 */
	public void decreaseSensitivity() {
		sensitivity -= 2;
		if (sensitivity < 1)
			sensitivity = 1;
	}

	/**
	 * Take a screenshot
	 * 
	 */
	public void takeScreenshot() {
		glViewer.screenshot();
	}

	/**
	 * we probably want to quit in some way
	 * 
	 */
	public void quit() {
		System.exit(0);
	}

	/**
	 * allow or disallow the user to move in the z-dimension
	 * 
	 * @param enable
	 *            true to allow movement along the z-axis
	 */
	public void setFreeMove(String enable) {
		if (enable.equals("on"))
			freeMove = true;
		else if (enable.equals("off"))
			freeMove = false;
	}

}
