/**
 * 
 */
package viewer;

import javax.media.opengl.*;
import javax.media.opengl.glu.*;
import com.sun.opengl.util.*;

import java.awt.Dimension;
import java.io.*;

import model.*;
import java.util.*;

/**
 * Displays the actual GL drawings and takes care of all GL needs for the
 * viewer.
 * 
 * @author Henry Gross
 * @author Ed Williams
 */
public class GLViewer implements GLEventListener {
	private Viewer viewer;

	private GLCanvas canvas;

	private GLAutoDrawable draw;

	private GL gl;

	private GLU glu;

	private GLUT glut;

	private Model map;

	public Lights lights;

	private int brightness;

	private int brightnessAmountChanged;

	private double[] cam;

	private double fov;

	private boolean textures;

	private boolean is2D;

	private boolean screenshot;

	private boolean changeLights;

	private boolean forceRenderNext = false;

	private boolean disableLights = false;

	private boolean enableLights = false;

	/**
	 * create a new instance of GLViewer and set up some basic variables
	 */
	public GLViewer(Viewer viewer) {
		this.viewer = viewer;
		canvas = new GLCanvas();
		cam = new double[] { 0, 0, 10000, 90, -85 };
		is2D = false;
		textures = true;

		setFov(80);

	}

	public void setForceRenderNext(boolean f) {
		forceRenderNext = f;
	}

	/**
	 * @return GLVanvas the canvas to be added to the frame
	 */
	public GLCanvas getCanvas() {
		return canvas;
	}

	/**
	 * set the internal map to the map that we want to view
	 * 
	 * @param map
	 *            the map that we will be viewing
	 */
	public void setMap(Model map) {
		this.map = map;
		if (gl != null)
			this.map.setGL(gl);
		if (glu != null)
			this.map.setGLU(glu);
		if (glut != null)
			this.map.setGLUT(glut);
	}

	public void setMapSameGL(Model map) {
		this.map = map;
	}

	/**
	 * start all of the GL code this is seperate since we might want to do some
	 * setup before we actually do this
	 */
	public void startGL() {
		canvas.addGLEventListener(this);
	}

	/**
	 * initialize all of the GL stuff
	 */
	public void init(GLAutoDrawable drawable) {
		draw = drawable;
		gl = drawable.getGL();
		glu = new GLU();
		glut = new GLUT();

		lights = new Lights(this, gl);

		lights.init();
		gl.glDepthFunc(GL.GL_LEQUAL);
		gl.glEnable(GL.GL_DEPTH_TEST);

		// added to normalize all the normals (glMultMatrix scaling affects
		// normals)
		gl.glEnable(GL.GL_NORMALIZE);

		gl.glClearColor(1.0f, 1.0f, 1.0f, 1.0f);
		gl.glClear(GL.GL_COLOR_BUFFER_BIT | GL.GL_DEPTH_BUFFER_BIT);

		gl.glMatrixMode(GL.GL_PROJECTION);
		gl.glLoadIdentity();
		glu.gluPerspective(fov, 1, 1, 5000);

		double lookX = cam[0] + Math.cos(Math.toRadians(cam[3]));
		double lookY = cam[1] + Math.sin(Math.toRadians(cam[3]));
		double lookZ = cam[2] + Math.sin(Math.toRadians(cam[4]));

		glu.gluLookAt(cam[0], cam[1], cam[2], lookX, lookY, lookZ, 0, 0, 1);

		gl.glMatrixMode(GL.GL_MODELVIEW);
		map.setGL(gl);
		map.setGLU(glu);
		map.setGLUT(glut);
		map.drawModel(true);

		draw();
	}

	/**
	 * draw what we want to draw
	 */
	public void display(GLAutoDrawable drawable) {
		float x = canvas.getWidth();
		float y = canvas.getHeight();

		gl.glClearColor(1.0f, 1.0f, 1.0f, 1.0f);
		gl.glClear(GL.GL_COLOR_BUFFER_BIT | GL.GL_DEPTH_BUFFER_BIT);

		gl.glMatrixMode(GL.GL_PROJECTION);
		gl.glLoadIdentity();
		glu.gluPerspective(fov, x / y, 1, 50000);

		double xy = cam[3];
		double z = cam[4];

		double plane = Math.cos(Math.toRadians(z));
		double lookX = cam[0] + Math.cos(Math.toRadians(xy)) * plane;
		double lookY = cam[1] + Math.sin(Math.toRadians(xy)) * plane;
		double lookZ = cam[2] + Math.sin(Math.toRadians(z));

		glu.gluLookAt(cam[0], cam[1], cam[2], lookX, lookY, lookZ, 0, 0, 1);

		if (is2D) {
			gl.glLoadIdentity();
			gl.glOrtho(-10000.0f, 10000.0f, -10000.0f, 10000.0f, -1.0f, 1.0f);
		}

		gl.glMatrixMode(GL.GL_MODELVIEW);

		map.drawModel(forceRenderNext);
		forceRenderNext = false;

		ArrayList<Double[]> people = viewer.getPeople();
		for (int i = 0; i < people.size(); i++) {
			Double[] pos = people.get(i);
			gl.glTranslated(pos[0], pos[1], pos[2]);
			gl.glColor3f(0.0f, 0.0f, 0.0f);
			glut.glutSolidSphere(1000, 10, 10);
			gl.glLoadIdentity();
		}

		if (screenshot) {
			takeScreenshot();
			screenshot = false;
		}
		if (changeLights) {
			lights.adjustDiffuse(brightnessAmountChanged);
			gl.glLightfv(GL.GL_LIGHT0, GL.GL_DIFFUSE, lights.light_diffuse, 0);
			gl.glLightfv(GL.GL_LIGHT1, GL.GL_DIFFUSE, lights.light_diffuse, 0);
			gl.glLightfv(GL.GL_LIGHT2, GL.GL_DIFFUSE, lights.light_diffuse, 0);
			gl.glLightfv(GL.GL_LIGHT0, GL.GL_DIFFUSE, lights.light_diffuse, 0);
			gl.glLightfv(GL.GL_LIGHT1, GL.GL_DIFFUSE, lights.light_diffuse, 0);
			gl.glLightfv(GL.GL_LIGHT2, GL.GL_DIFFUSE, lights.light_diffuse, 0);
			gl.glLightfv(GL.GL_LIGHT0, GL.GL_AMBIENT, lights.light_ambient, 0);
			gl.glLightfv(GL.GL_LIGHT1, GL.GL_AMBIENT, lights.light_ambient, 0);
			gl.glLightfv(GL.GL_LIGHT2, GL.GL_AMBIENT, lights.light_ambient, 0);
			gl.glLightfv(GL.GL_LIGHT0, GL.GL_AMBIENT, lights.light_ambient, 0);
			gl.glLightfv(GL.GL_LIGHT1, GL.GL_AMBIENT, lights.light_ambient, 0);
			gl.glLightfv(GL.GL_LIGHT2, GL.GL_AMBIENT, lights.light_ambient, 0);
			changeLights = false;

		}
		if (disableLights) {
			gl.glDisable(GL.GL_LIGHTING);
			disableLights = false;
		}
		if (enableLights) {
			gl.glEnable(GL.GL_LIGHTING);
			enableLights = false;
		}
	}

	public void displayChanged(GLAutoDrawable drawable, boolean modeChanged,
			boolean deviceChanged) {
	}

	public void reshape(GLAutoDrawable drawable, int x, int y, int width,
			int height) {
	}

	/**
	 * returns the camera array
	 * 
	 * @return the camera array
	 */
	public double[] getCam() {
		double[] newCam = new double[cam.length];
		System.arraycopy(cam, 0, newCam, 0, cam.length);
		return newCam;
	}

	/**
	 * Sets the camera array to the the camera array gioven
	 * 
	 * @param cam
	 *            the camera array to set the camera to
	 */
	public void setCam(double[] cam) {
		this.cam = cam;
	}

	/**
	 * Set the field of vision to the specified angle
	 * 
	 * @param fov
	 *            the fov angle we want
	 */
	public void setFov(double fov) {
		this.fov = fov;
	}

	/**
	 * redraw the screen
	 */
	public void draw() {
		draw.display();
	}

	/**
	 * set whether to display in 3D or 2D
	 * 
	 * @param b
	 *            true if to display in 3D
	 */
	public void set3D(boolean b) {
		is2D = !b;
	}

	/**
	 * Set whether or not to display the textures
	 * 
	 * @param b
	 *            true if we wnat to display textures
	 */
	public void showTextures(boolean b) {
		textures = b;
		map.texturesOn(b);
		draw.display();
	}

	/**
	 * Change the brightness of the lights
	 * 
	 * @param i
	 *            the intensity of the lights
	 */
	public void changeBrightness(int i) {

		// gl = draw.getGL();
		brightnessAmountChanged = i - brightness;

		brightness = i;
		changeLights = true;
		draw.display();

	}

	/**
	 * we changed the lights, so update gl with the new lights
	 * 
	 * @param b
	 *            true if we changed lights
	 */
	public void updateLights(boolean b) {
		if (b)
			changeLights = true;
		else
			changeLights = false;
		draw.display();
	}

	/**
	 * turn lights on or off
	 * 
	 * @param b
	 *            true to turn lights on
	 */
	public void setLights(boolean b) {
		if (!b)
			disableLights = true;
		else
			enableLights = true;

		draw.display();
	}

	/**
	 * Return the GL object
	 * 
	 * @return the gl object
	 */
	public GL getGL() {
		return gl;
	}

	/**
	 * request that the gl receives focus
	 */
	public void requestFocus() {
		canvas.requestFocus();
	}

	/**
	 * actually take a screenshot
	 */
	public void takeScreenshot() {
		Dimension dim = canvas.getSize();
		Date date = new Date();
		String stamp = Long.toString(date.getTime());
		File file = new File(Viewer.getDataPath() + "screenshots/" + stamp
				+ ".tga");
		try {
			file.createNewFile();
			Screenshot.writeToTargaFile(file, dim.width, dim.height);
		} catch (IOException e) {
			e.printStackTrace(System.out);
		}
	}

	/**
	 * take a screenshot have to do this messy hack because of glContexts not
	 * being active on threads or some such thing
	 */
	public void screenshot() {
		screenshot = true;
		draw();
	}

	/**
	 * return if textures are on
	 * 
	 * @return true is the textures are being displayed
	 */
	public boolean getTextureState() {
		return textures;
	}
}
