/**
 * 
 */
package viewer;

import java.awt.*;
import java.awt.event.*;
import java.io.*;

/**
 * Method of input and output to be connected to the viewer. Ideally would
 * replace System.in and System.out, which would be useful for fullscreen
 * 
 * @author Henry Gross
 */
public class Console {
	private PrintStream out;

	private ConsoleInput in;

	private Panel console;

	private List consoleOut;

	private TextField consoleIn;

	private FakeConsoleInput fakeIn;

	private Interaction interaction;

	/**
	 * An output stream that is tied to the programso that it is free from the
	 * terminal
	 * 
	 * @author Henry
	 */
	private class ConsoleOutput extends OutputStream {
		List consoleOut;

		/**
		 * create a new output stream and get a List to "print" to
		 */
		public ConsoleOutput(List consoleOut) {
			this.consoleOut = consoleOut;
		}

		/**
		 * write the stuff to the consoleOut list
		 */
		@Override
		public void write(int b) throws IOException {
			// TODO break line if over x characters
			Byte bt = Byte.decode(Integer.toString(b));
			char ch = (char) bt.byteValue();
			if (ch == 10)
				consoleOut.add("");
			else if (ch >= 32) {
				String line = consoleOut.getItem(consoleOut.getItemCount() - 1);
				consoleOut.replaceItem(line + Character.toString(ch),
						consoleOut.getItemCount() - 1);
			}
			if (consoleOut.getItemCount() > 4)
				consoleOut.remove(0);
		}
	}

	/**
	 * An input stream for the program to use to free it from the terminal
	 * 
	 * @author Henry
	 */
	private class ConsoleInput extends InputStream implements ActionListener {
		// TODO: make this work like System.in
		TextField consoleIn;

		String st;

		/**
		 * Create an input stream and get a TextField to take input from
		 * 
		 * @param consoleIn
		 *            the TextField to take input from
		 */
		public ConsoleInput(TextField consoleIn) {
			this.consoleIn = consoleIn;
			st = new String();
		}

		/**
		 * read the next byte from consoleIn
		 */
		@Override
		public synchronized int read() {
			waitForInput();
			char ch = st.charAt(0);
			st = st.substring(1);
			return ch;
			// System.err.println("char");
			// return 0;
		}

		/**
		 * There won't always be input, so we must be paitent and wait for it
		 * 
		 */
		private void waitForInput() {
			try {
				while (consoleIn.getText().length() == 0 && st.length() == 0) {
					System.err.println("waiting");
					wait();
					System.err.println("done waiting");
				}
			} catch (InterruptedException e) {
			}
		}

		/**
		 * We do need to know however when there is an action performed so we
		 * know we can read
		 */
		public synchronized void actionPerformed(ActionEvent e) {
			st = consoleIn.getText();
			consoleIn.setText("");
			notifyAll();
		}
	}

	/**
	 * Since the input stream isn't fully working, we will have to do this to
	 * get input for now
	 * 
	 * @author Henry
	 */
	private class FakeConsoleInput implements ActionListener, KeyListener {
		TextField consoleIn;

		Interaction interaction;

		String st;

		/**
		 * create a way for the user to enter commands
		 * 
		 * @param consoleIn
		 *            the textField to use for input
		 */
		public FakeConsoleInput(TextField consoleIn) {
			this.consoleIn = consoleIn;
			st = new String();
		}

		/**
		 * It would be good to have someplace to send the stuff that is typed in
		 * 
		 * @param interaction
		 *            the interaction instance to send stuff to
		 */
		public void attachInteraction(Interaction interaction) {
			this.interaction = interaction;
		}

		/**
		 * act on the input when the user hits "enter"
		 */
		public void actionPerformed(ActionEvent arg0) {
			st = consoleIn.getText();
			consoleIn.setText("");
			interaction.execute(st);
		}

		/**
		 * We do want a way of closing this if we decide we don't want to see it
		 */
		public void keyPressed(KeyEvent e) {
			int keyCode = e.getKeyCode();
			String keyText = KeyEvent.getKeyText(keyCode);
			if (keyText == "Back Quote") {
				interaction.toggleConsole();
				e.consume();
			} else if (keyText == "Escape") {
				interaction.toggleOptions();
				e.consume();
			}
		}

		public void keyReleased(KeyEvent arg0) {
		}

		public void keyTyped(KeyEvent arg0) {
		}

	}

	/**
	 * Provides a way for the user to send/receive commands to/from the program.
	 * Meant to provide a better interface than just the standard terminal
	 * stdin/out
	 * 
	 */
	public Console() {
		console = new Panel(new BorderLayout());
		consoleOut = new List();
		consoleOut.add("");
		console.add(consoleOut, BorderLayout.CENTER);
		consoleIn = new TextField();
		console.add(consoleIn, BorderLayout.SOUTH);

		// TODO set properties

		ConsoleOutput consoleOutput = new ConsoleOutput(consoleOut);
		out = new PrintStream(consoleOutput);
		in = new ConsoleInput(consoleIn);
		// consoleIn.addActionListener(in);
		fakeIn = new FakeConsoleInput(consoleIn);
		consoleIn.addActionListener(fakeIn);
		console.addKeyListener(fakeIn);
		consoleIn.addKeyListener(fakeIn);
		consoleOut.addKeyListener(fakeIn);
	}

	/**
	 * Gets the cosnole object so it can be attached to the main progran
	 * 
	 * @return the entire console
	 */
	public Panel getConsole() {
		return console;
	}

	/**
	 * Gets the output stream so that it can be attached to the main program
	 * 
	 * @return the output stream
	 */
	public PrintStream getOut() {
		return out;
	}

	/**
	 * Gets the input stream so that it can be attached to the main program
	 * 
	 * @return the input stream
	 */
	public InputStream getIn() {
		return in;
	}

	/**
	 * Give consoleIn somewhere to send the commands to
	 * 
	 * @param interaction
	 */
	public void attachInteraction(Interaction interaction) {
		this.interaction = interaction;
		fakeIn.attachInteraction(this.interaction);
	}

	/**
	 * Show or hide the console
	 * 
	 * @param b
	 *            true to display the console
	 */
	public void setVisible(boolean b) {
		console.setVisible(b);
	}

	/**
	 * Get the visibility of the console
	 * 
	 * @return true if the console is visible
	 */
	public boolean isVisible() {
		return console.isVisible();
	}

	/**
	 * Request that the text field receive focus, since that would be nice
	 */
	public void requestFocus() {
		consoleIn.requestFocus();
	}

}
