package server;

import java.io.*;
import java.net.*;

/**
 * A general client class for connecting to a server instance
 * 
 * @author Henry
 * 
 */
public abstract class Client extends Thread {
	private MulticastSocket socket = null;

	private DatagramPacket packet = null;

	private InetAddress serverAddress = null;

	private InetAddress address = null;

	private InetAddress group = null;

	private String groupName = null;

	private final int DEFAULT_PORT = 2006;

	private final int SERVER_PORT = 2007;

	private int localPort = DEFAULT_PORT;

	private boolean connected = false;

	/**
	 * connect to a server and open the default port
	 * 
	 * @param address
	 *            the address of the server it should send data to
	 * @throws IOException
	 */
	public void connect(String address) throws IOException {
		connect(address, DEFAULT_PORT);
	}

	/**
	 * connect to a server and specifies a port to open up
	 * 
	 * @param address
	 *            the address of the server it should send data to
	 * @param localPort
	 *            the local port to open
	 * @return True if the connection was successful, False otherwise
	 * @throws IOException
	 */
	public boolean connect(String address, int localPort) throws IOException {
		this.address = InetAddress.getByName(InetAddress.getLocalHost()
				.getHostAddress());
		this.localPort = localPort;
		socket = new MulticastSocket(this.localPort);
		if (address.equals("127.0.0.1"))
			address = InetAddress.getLocalHost().getHostAddress();
		serverAddress = InetAddress.getByName(address);

		if (serverAddress.isReachable(1000)) {

			byte[] buf = new byte[512];
			buf = Common.encodeStringPacket("/connect");

			packet = new DatagramPacket(buf, buf.length, serverAddress,
					SERVER_PORT);
			socket.send(packet);

			buf = new byte[512];
			packet = new DatagramPacket(buf, buf.length);
			socket.receive(packet);

			groupName = Common.decodeStringPacket(packet.getData());
			group = InetAddress.getByName(groupName);
			socket.joinGroup(group);
			connected = true;
			return true;
		} else
			return false;
	}

	/**
	 * be ready to receive packets
	 */
	@Override
	public void run() {
		while (true) {
			byte[] buf = new byte[512];
			packet = new DatagramPacket(buf, buf.length);
			try {
				socket.receive(packet);
			} catch (IOException e) {
			}
			if (packet.getAddress().equals(serverAddress))
				react(Common.decodeStringPacket(packet.getData()));
		}
	}

	/**
	 * send a packet receive a packet
	 */
	public void ping() {
		byte[] buf = new byte[512];
		buf = Common.encodeStringPacket("/ping");

		packet = new DatagramPacket(buf, buf.length, serverAddress, SERVER_PORT);
		try {
			socket.send(packet);
		} catch (IOException e) {
		}
		System.out.println("packet sent");

		buf = new byte[512];
		packet = new DatagramPacket(buf, buf.length);
		try {
			socket.receive(packet);
		} catch (IOException e) {
		}
		System.out.println("packet received");
	}

	/**
	 * disconnect from the server
	 * 
	 * @throws IOException
	 */
	public void disconnect() throws IOException {
		socket.leaveGroup(group);
		socket.close();
		connected = false;
	}

	/**
	 * send a string to the server
	 * 
	 * @param cmd
	 *            the string to send to the server
	 * @throws IOException
	 */
	public void sendString(String cmd) throws IOException {
		byte[] buf = new byte[512];
		buf = Common.encodeStringPacket(cmd);
		packet = new DatagramPacket(buf, buf.length, serverAddress, SERVER_PORT);
		socket.send(packet);
	}

	/**
	 * Get the address of the client
	 * 
	 * @return the string form of the address of the client
	 */
	public String getAddress() {
		return address.toString().substring(1);
	}

	/**
	 * Check if the clinet is connected
	 * 
	 * @return true if the client is connected
	 */
	public boolean isConnected() {
		return connected;
	}

	/**
	 * function to be called when the client receives a packet from the server
	 * to be implemented by a specific client
	 * 
	 * @param cmd
	 *            the response from the server
	 */
	protected abstract void react(String cmd);
}
