package preditor;
import model.*;

/**
 * Wrapper class for either a Model or a ComplexObject.
 * Used in the CMStack so that either one may be selected.
 * @author chasmand
 * @version 1/08/07
 */

public class SelectionType {
	private Model model = null;				//May hold a Model.
	private ComplexObject complexObject = null;	//Otherwise, may hold a ComplexObject.
	private boolean isModel;			//If false, assumed to be ComplexObject.
	
	/**
	 * Constructs a new SelectionType instance holding a Model object.
	 * @param m	The Model to be wrapped up
	 */
	public SelectionType(Model m) {
		model = m;
		complexObject = null;
		isModel = true;
	}
	
	/**
	 * Constructs a new SelectionType instance holding a ComplexObject object.
	 * @param co The ComplexObject to be wrapped up
	 */
	public SelectionType(ComplexObject co) {
		complexObject = co;
		model = null;
		isModel = false;
	}
	
	/**
	 * Accessor for information about type of enclosed object.
	 * @return true True if Model, False if ComplexObject
	 */
	public boolean isModel() {
		return isModel;
	}
	
	/**
	 * Accessor for information about type of enclosed object.
	 * @return true True if ComplexObject, False if model
	 */
	public boolean isComplexObject() {
		return !isModel;
	}
	
	/**
	 * Accessor for stored object.
	 * @return stored Object
	 */
	public Object getContents() {
		Object r;
		if (isModel) {
			r = model;
		} else {
			r = complexObject;
		}		
		return r;
	}
	
	/**
	 * Accessor for stored Model.
	 * @return stored Model, or null
	 */
	public Model getModel() {
		return model;
	}
	
	/**
	 * Accessor for stored ComplexObject.
	 * @return stored ComplexObject, or null
	 */
	public ComplexObject getComplexObject() {
		return complexObject;
	}
	
	/**
	 * Accesses transformation of this SelectionType.
	 * @return transformation
	 */
	public Transformation getTransformation() {
		if (isModel)
			return model.getTransformation();
		else
			return complexObject.getTransformation();
	}
	
	/**
	 * Mutates this SelectionType's transformation.
	 * @param t	new transformation
	 */
	public void setTransform(Transformation t) {
		if (isModel)
			model.setTransform(t);
		else
			complexObject.setTransform(t);
	}
	
	/**
	 * Accesses this SelectionType's alias.
	 * @return alias
	 */
	public String getAlias() {
		if (isModel)
			return model.getAlias();
		else
			return complexObject.getAlias();
	}
	
	
	/**
	 * Mutates this SelectionType's alias.
	 * @param alias	new alias
	 */
	public void setAlias(String alias) {
		if (isModel)
			model.setAlias(alias);
		else
			complexObject.setAlias(alias);
	}
	
}
