package preditor;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.event.*;
import java.io.*;
import java.text.NumberFormat;
import java.text.ParseException;
import java.util.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.DefaultFormatterFactory;
import javax.swing.text.NumberFormatter;
import javax.swing.tree.*;

import preditor.CMStack;
import preditor.SelectionType;

import model.*;
import model.ObjectType.ObjType;
import viewer.*;

/**
 * Editor component of the Campus Model Group Comps, Fall 2006/Winter 2007.
 * 
 * Questions/comments? E-mail deborah.chasman@gmail.com. 
 * 
 * @author Debbie Chasman
 * @author with some assistance from Henry Gross, Ed Williams, and Paul Wilmes.
 */

/*
 * This code was edited or generated using CloudGarden's Jigloo
 * SWT/Swing GUI Builder, which is free for non-commercial
 * use. If Jigloo is being used commercially (ie, by a corporation,
 * company or business for any purpose whatever) then you
 * should purchase a license for each developer using Jigloo.
 * Please visit www.cloudgarden.com for details.
 * Use of Jigloo implies acceptance of these licensing terms.
 * A COMMERCIAL LICENSE HAS NOT BEEN PURCHASED FOR
 * THIS MACHINE, SO JIGLOO OR THIS CODE CANNOT BE USED
 * LEGALLY FOR ANY CORPORATE OR COMMERCIAL PURPOSE.
 */
public class Preditor extends javax.swing.JFrame implements ActionListener, 
													TreeSelectionListener,
													ChangeListener,
													MouseListener {


	/*
	 * Constants
	 */
	//min and max for the sliders
	/**
	 * Minimum for translation slider.
	 */
	private final int TRANS_MIN = -1000;	
	/**
	 * Maximum for translation slider.
	 */
	private final int TRANS_MAX = 1000;
	/**
	 * Initial value for translation slider.
	 */
	private final int TRANS_INIT = 0;
	/**
	 * Minimum for rotation slider.
	 */
	private final int ROT_MIN = 0;
	/**
	 * Maximum for rotation slider.
	 */
	private final int ROT_MAX = 360;
	/**
	 * Initial value for rotation slider.
	 */
	private final int ROT_INIT = 0;
	
	//Field and slider have different bounds
	/**
	 * Minimum for scale field.
	 */
	private final double SCALE_F_MIN = .1;		//field min
	/**
	 * Maximum for scale field.
	 */
	private final double SCALE_F_MAX = 10;		//field max
	/**
	 * Minimum for scale slider.
	 */
	private final int SCALE_S_MIN = 1;			//slider min
	/**
	 * Maximum for scale slider.
	 */
	private final int SCALE_S_MAX = 100;		//slider max
	/**
	 * Initial value for scale slider.
	 */
	private final int SCALE_S_INIT = 10;		//slider init

	//Limits on the transform-to boxes
	/**
	 * Marks the minimum for each axis in the world.
	 */
	private final int TO_MIN = -20000; 
	/**
	 * Maximum for each axis in the world.
	 */
	private final int TO_MAX = 20000;

	//In place of an enum...
	private final int X = 0;
	private final int Y = 1;
	private final int Z = 2;

	//Used in tree selection
	/**
	 * Used in updateTree() to choose an origin of updates.
	 */
	private final int FROM_PARENT = 0;
	/**
	 * Used in updateTree() to choose an origin of updates.
	 */
	private final int FROM_HERE = 1;

	/*
	 * GUI Vars
	 */
	//Set up menu bar at top of screen
	private JMenuBar topMenuBar;	
	private JMenu fileMenu;				//File pull-down
	private JMenuItem saveAsMenuItem;
	private JMenuItem saveMenuItem;
	private JMenuItem openFileMenuItem;
	private JMenuItem newFileMenuItem;	
	private JMenuItem exitMenuItem;
	private JSeparator fileMenuSeparator;

	private JMenu editMenu;				//Edit pull-down
	private JMenuItem copyMenuItem;
	private JMenuItem cutMenuItem;
	private JMenuItem deleteMenuItem;
	private JMenuItem pasteMenuItem;	
	private JMenuItem addObjectMenuItem;
	private JMenuItem addComplexObjectMenuItem;
	private JMenuItem addSubModelMenuItem;
	private JSeparator editMenuSeparator;

	private JMenu helpMenu;				//Help pull-down
	private JMenuItem helpMenuItem;

	//Editor tab components
	private JTabbedPane Tabs;
	private JPanel editorTab;

	private JPanel editButtonPanel;
	private JPanel visualsPanel;
	private JButton textureButton;
	private JButton colorButton;
	private JButton aliasButton;
	private JPanel transformationPanel;

	//Radio Buttons!
	private JPanel transformationButtonPanel;
	private JRadioButton rotationRadioButton;
	private JRadioButton translationRadioButton;
	private JRadioButton scalingRadioButton;
	private ButtonGroup transformationButtonGroup;

	//Constants to designate which transformation mode was chosen
	private final int TRANS = 0;
	private final int ROTATE = 1;
	private final int SCALE = 2;
	private int whichTransformation = 0;	//Translation by default

	//Sliders!	
	private JPanel sliderPanel;
	private JSlider xSlider;
	private JSlider ySlider;
	private JSlider zSlider;
	//We want to disable transformation when switching transformation
	//types (because it has problems with scale)
	private boolean disableTransformation = false;	

	//Text fields
	private JFormattedTextField xField, yField, zField;
	private JSpinner toXField, toYField, toZField;
	NumberFormat numberFormat;
	NumberFormatter transFormatter, rotationFormatter, scaleFormatter;
	NumberFormatter transAndRotateToFormatter, scaleToFormatter;

	//Object tree components (in editor tab)
	private JLabel treeLabel;
	private JTree modelTree;
	private JScrollPane treeScrollPane;

	//Current attribute panel components
	//Shows current attributes (position, color, rotation, scale) 
	//of selected object
	private JTextField selectionField;
	private JTextField typeField;
	private JTextField nameField;
	private JTextField positionField;
	private JTextField colorField;
	private JTextField scaleField;
	private JTextField rotationField;
	private JLabel currentAttrLabel;
	private JPanel attributePanel;

	//The model itself!
	private Model loadedModel;		//The currently loaded model
	private String loadedFile;		//Name of open file/directory

	//Pertaining to current selection
	
	/**
	 * IDList of currently selected model.
	 */
	private IDList currentIDList = null;
	/**
	 * Objects belonging to currently selected model.
	 */
	private ArrayList<ComplexObject> currentObjectList = null; 
	/**
	 * Submodels belonging to currently loaded file.
	 */
	private ArrayList<Model> currentSubModelList = null;
	/**
	 * Currently selected object.
	 */
	private ComplexObject selectedObject = null;
	/**
	 * Currently selected model.
	 */
	private Model selectedModel = null;
	/**
	 * Is current selection a model?
	 */
	private boolean isSelectedModel;
	/**
	 * Keeps track of selections.
	 */
	private CMStack<SelectionType> selectionStack = null;	//Stack of selections (top is current selection)

	/**
	 * The clipboard, which can hold a Model or an Object.
	 */
	private SelectionType clipboard = null;

	/**
	 * Are the sliders in use?
	 */
	private boolean sliding = false;
	/**
	 * Temporarily hold a transformation when sliders are in use.
	 */
	private Transformation tempTrans;

	/**
	 * The viewer!
	 */
	private Viewer previewer;

	/**
	 * Displays the GUI.
	 */
	public static void main(String[] args) {		
		Preditor inst = new Preditor();
	}

	/**
	 * Constructs a Preditor object, initializes it, and displays it.
	 */ 
	public Preditor() {

		//call on the constructor of the parent class (JFrame)
		super();

		//Make a viewer
		previewer = new Viewer(false, false);

		//Load the model, set up GUI, etc
		initModelFirstTime();		
		initGUI();
		displayAttributes();

		this.setVisible(true);
		this.setDefaultCloseOperation(EXIT_ON_CLOSE);
	}


	/**
	 * Sets up initial GUI.
	 */
	private void initGUI() {
		try {
			this.setTitle("Preditor");
			//Set up tabbed pane 
			//(Maybe some day we'll have a use for a second tab?)

			Tabs = new JTabbedPane();
			getContentPane().add(Tabs);
			Tabs.setPreferredSize(new Dimension(1000, 300));

			editorTab = new JPanel();
			Tabs.addTab("Editor", null, editorTab, null);
			editorTab.setLayout(null);

			//Model tree inside the editor tab
			treeLabel = new JLabel();
			editorTab.add(treeLabel);
			treeLabel.setText("Model Tree");
			treeLabel.setBounds(7, 0, 112, 28);

			treeScrollPane = new JScrollPane();
			editorTab.add(treeScrollPane);
			treeScrollPane.setBounds(14, 21, 182, 196);

			//Create the tree
			createTree();

			/*
			 * The attribute panel holds a display of the current attributes
			 * of the selected component of the model 
			 */
			attributePanel = new JPanel();
			editorTab.add(attributePanel);
			attributePanel.setBounds(700, 15, 210, 300);
			attributePanel.setLayout(null);

			currentAttrLabel = new JLabel();
			attributePanel.add(currentAttrLabel);
			currentAttrLabel.setText("Current Selections:");
			currentAttrLabel.setBounds(7, 7, 200, 23);

			selectionField = new JTextField();
			attributePanel.add(selectionField);
			selectionField.setText("In Focus:");
			selectionField.setBounds(7, 30, 200, 30);
			selectionField.setEditable(false);

			nameField = new JTextField();
			attributePanel.add(nameField);
			nameField.setText("Object Name:");
			nameField.setBounds(7, 60, 200, 30);
			nameField.setEditable(false);

			typeField = new JTextField();
			attributePanel.add(typeField);
			typeField.setText("Type:");
			typeField.setBounds(7, 90, 200, 30);
			typeField.setEditable(false);

			positionField = new JTextField();
			attributePanel.add(positionField);
			positionField.setText("Position: (x, y, z)");
			positionField.setBounds(7, 120, 200, 30);
			positionField.setEditable(false);

			rotationField = new JTextField();
			attributePanel.add(rotationField);
			rotationField.setText("Rotation: (x, y, z)");
			rotationField.setBounds(7, 150, 200, 30);
			rotationField.setEditable(false);

			scaleField = new JTextField();
			attributePanel.add(scaleField);
			scaleField.setText("Scaling: (x, y, z)");
			scaleField.setBounds(7, 180, 200, 30);
			scaleField.setEditable(false);

			colorField = new JTextField();
			attributePanel.add(colorField);
			colorField.setText("Color: (r, g, b, a)");
			colorField.setBounds(7, 210, 200, 30);
			colorField.setEditable(false);

			/*
			 * The edit button panel holds buttons and input fields
			 * with which the user may transform the selected object.
			 */
			editButtonPanel = new JPanel();
			editorTab.add(editButtonPanel);
			editButtonPanel.setBounds(210, 20, 500, 300);
			editButtonPanel.setLayout(null);

			//The transformation panel holds input areas
			//for new transformations
			transformationPanel = new JPanel();
			editButtonPanel.add(transformationPanel);
			transformationPanel.setLayout(null);
			transformationPanel.setBounds(0, 0, 490, 150);

			//Add radio button panel
			transformationButtonGroup = new ButtonGroup();
			transformationButtonPanel = new JPanel();
			transformationPanel.add(transformationButtonPanel);
			transformationButtonPanel.setBounds(0,0,490,30);	

			//Make radio buttons
			translationRadioButton = new JRadioButton("Translate", true);
			transformationButtonGroup.add(translationRadioButton);
			transformationButtonPanel.add(translationRadioButton);
			translationRadioButton.addActionListener(this);

			rotationRadioButton = new JRadioButton("Rotate", false);
			transformationButtonGroup.add(rotationRadioButton);
			transformationButtonPanel.add(rotationRadioButton);
			rotationRadioButton.addActionListener(this);

			scalingRadioButton = new JRadioButton("Scale", false);
			transformationButtonGroup.add(scalingRadioButton);
			transformationButtonPanel.add(scalingRadioButton);
			scalingRadioButton.addActionListener(this);

			//Set up formatters and sliders
			setUpFormattersAndSliders();

			//The visuals panel holds color and texture buttons
			visualsPanel = new JPanel();
			editButtonPanel.add(visualsPanel);
			visualsPanel.setBounds(0, 150, 490, 50);						
			colorButton = new JButton();
			visualsPanel.add(colorButton);
			colorButton.setText("Choose color?");
			colorButton.addActionListener(this);

			aliasButton = new JButton();
			visualsPanel.add(aliasButton);
			aliasButton.setText("Change name");
			aliasButton.addActionListener(this);

			textureButton = new JButton();
			visualsPanel.add(textureButton);
			textureButton.setText("Add texture?");
			textureButton.addActionListener(this);

			/*
			 * Sets up the menu bar at the top of the window.
			 */
			topMenuBar = new JMenuBar();
			setJMenuBar(topMenuBar);
			//Set up file pull-down
			fileMenu = new JMenu();
			topMenuBar.add(fileMenu);
			fileMenu.setText("File");
			{
				newFileMenuItem = new JMenuItem();
				fileMenu.add(newFileMenuItem);
				newFileMenuItem.setText("New model");
				newFileMenuItem.addActionListener(this);

				openFileMenuItem = new JMenuItem();
				fileMenu.add(openFileMenuItem);
				openFileMenuItem.setText("Open");
				openFileMenuItem.addActionListener(this);

				saveMenuItem = new JMenuItem();
				fileMenu.add(saveMenuItem);
				saveMenuItem.setText("Save");
				saveMenuItem.addActionListener(this);

				saveAsMenuItem = new JMenuItem();
				fileMenu.add(saveAsMenuItem);
				saveAsMenuItem.setText("Save As ...");
				saveAsMenuItem.addActionListener(this);
				fileMenuSeparator = new JSeparator();
				fileMenu.add(fileMenuSeparator);

				exitMenuItem = new JMenuItem();
				fileMenu.add(exitMenuItem);
				exitMenuItem.setText("Exit");
				exitMenuItem.addActionListener(this);
			}

			//Set up Edit pull-down
			editMenu = new JMenu();
			topMenuBar.add(editMenu);
			editMenu.setText("Edit");
			{
				cutMenuItem = new JMenuItem();
				editMenu.add(cutMenuItem);
				cutMenuItem.setText("Cut");
				cutMenuItem.addActionListener(this);

				copyMenuItem = new JMenuItem();
				editMenu.add(copyMenuItem);
				copyMenuItem.setText("Copy");
				copyMenuItem.addActionListener(this);

				pasteMenuItem = new JMenuItem();
				editMenu.add(pasteMenuItem);
				pasteMenuItem.setText("Paste");
				pasteMenuItem.addActionListener(this);

				addObjectMenuItem = new JMenuItem();
				editMenu.add(addObjectMenuItem);
				addObjectMenuItem.setText("Add object");
				addObjectMenuItem.addActionListener(this);

				addComplexObjectMenuItem = new JMenuItem();
				editMenu.add(addComplexObjectMenuItem);
				addComplexObjectMenuItem.setText("Add complex object");
				addComplexObjectMenuItem.addActionListener(this);

				addSubModelMenuItem = new JMenuItem();
				editMenu.add(addSubModelMenuItem);
				addSubModelMenuItem.setText("Add sub-model");
				addSubModelMenuItem.addActionListener(this);

				editMenuSeparator = new JSeparator();
				editMenu.add(editMenuSeparator);

				deleteMenuItem = new JMenuItem();
				editMenu.add(deleteMenuItem);
				deleteMenuItem.setText("Delete");
				deleteMenuItem.addActionListener(this);
			}

			//Set up Help pull-down
			helpMenu = new JMenu();
			topMenuBar.add(helpMenu);
			helpMenu.setText("Help");
			{
				helpMenuItem = new JMenuItem();
				helpMenu.add(helpMenuItem);
				helpMenuItem.setText("Help");
				helpMenuItem.addActionListener(this);
			}

		} catch (Exception e) {
			e.printStackTrace();
		}

		//compress ("shrink-wrap") window to the smallest size
		this.pack();

	}

	/**
	 *  Instantiates the currentModel, either by creating a new 
	 *  model or by loading an existing one.
	 *  Called upon when editor is first opened; other times, 
	 *  will call on loadModel or createModel
	 */
	private void initModelFirstTime() {
		String input;
		int inputInt = -1;
		boolean choiceMade = false;
		boolean successfullyLoaded = false; //Whether or not model has successfully
		//Been loaded

		//open a dialog window (or terminal for now)
		while (successfullyLoaded == false || choiceMade == false) {
			input = JOptionPane.showInputDialog(null, "Would you like to (1) create a new model or (2) load an existing model?");

			if (input != null) {	
				try {
					inputInt = Integer.parseInt(input);
				} catch (NumberFormatException e) {					
				}

				if(inputInt == 1){
					successfullyLoaded = createModel(true);
					choiceMade = true;
				}			
				else if(inputInt == 2) {
					successfullyLoaded = loadModel(true);
					choiceMade = true;
				}			
			} else {
				System.exit(0);
			}
		}		
	}

	/**
	 * Manages the loading of a model directory.
	 * If it is not the first time, this method does not care whether or not
	 * the user actually chooses a directory. If it IS the first time, the user must
	 * choose a directory.
	 * Note that only directories may be opened. 
	 * @param firstTime true if called when program first opened
	 * @return true if model successfully loaded
	 */
	private boolean loadModel(boolean firstTime) {
		boolean methodSuccess = false;				//Model not chosen
		File file;
		File[] contents; //If the chosen file is a directory, this will contains
		//its contents
		JFileChooser fileChooser = new JFileChooser(Viewer.getDataPath());		
		fileChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);

		int choiceVal = fileChooser.showOpenDialog(Preditor.this);
		if (choiceVal == JFileChooser.APPROVE_OPTION) {	//Success

			//Set up pointers to loaded things
			file = fileChooser.getSelectedFile();

			//If it's a file, load just that file.
			if (file.isFile()) {
				loadedFile = file.getPath();
				loadedModel = new Model(file);	

				initVarOnLoad(); 						

				if (!firstTime) {
					createTree();
				}

				methodSuccess = true;
			} else if (file.isDirectory()) {
				loadedFile = file.getPath();
				loadedModel = new Model(file);
				loadedModel.setAlias(file.getName());

				contents = file.listFiles();
				if (contents != null) {            		            		
					//init vars
					initVarOnLoad();

					if (!firstTime) {
						createTree();			
					}            		
					methodSuccess = true;
				}
			}
		} else {				//Cancelled
			methodSuccess = false;
		}

		return methodSuccess;
	}

	/**
	 * 	Creates a new model directory.
	 *	
	 * 	@param firstTime true if called when program opened; 
	 * 		   false if called during program execution
	 * 	@return true if success, false if failure
	 */
	private boolean createModel(boolean firstTime){
		String filename = null;
		File newFile = null;
		boolean choiceSuccess = false;

		//get a filename
		filename = JOptionPane.showInputDialog("Please input a name for your new model directory:");

		if (filename != null) {		
			//always save as directory
			newFile = new File(Viewer.getDataPath() + "model/" + filename);
			choiceSuccess = newFile.mkdir();
		}

		if (!choiceSuccess) {
			JOptionPane.showMessageDialog(null, "Name already in use (or is otherwise problematic).");
			return false;
		}

		//Success?
		if (choiceSuccess) {
			System.out.println(filename);
			loadedFile = Viewer.getDataPath() + "model/" + filename;
			loadedModel = new Model(newFile);

			System.out.println("Now editing "+ loadedFile);			
			initVarOnLoad();			

			if (!firstTime) 
				createTree();
		}		
		return choiceSuccess;
	}

	/**
	 * Sets sliders and their accompanying fields
	 * based on which transformation is under consideration.
	 */
	private void setSliders() {
		float[] t;

		switch (whichTransformation) {
		case TRANS:
			xField.setFormatterFactory(new DefaultFormatterFactory(transFormatter));
			yField.setFormatterFactory(new DefaultFormatterFactory(transFormatter));
			zField.setFormatterFactory(new DefaultFormatterFactory(transFormatter));

			xField.setValue(new Integer(TRANS_INIT));
			yField.setValue(new Integer(TRANS_INIT));
			zField.setValue(new Integer(TRANS_INIT));

			if (selectedObject != null) {
				t = selectedObject.getTransformation().getTranslation();
				toXField.setValue(t[0]);
				toYField.setValue(t[1]);
				toZField.setValue(t[2]);
			}			
			xSlider.setMinimum(TRANS_MIN);
			xSlider.setMaximum(TRANS_MAX);			
			ySlider.setMinimum(TRANS_MIN);
			ySlider.setMaximum(TRANS_MAX);			
			zSlider.setMinimum(TRANS_MIN);
			zSlider.setMaximum(TRANS_MAX);

			break;

		case ROTATE:
			xField.setFormatterFactory(new DefaultFormatterFactory(rotationFormatter));
			yField.setFormatterFactory(new DefaultFormatterFactory(rotationFormatter));
			zField.setFormatterFactory(new DefaultFormatterFactory(rotationFormatter));

			xField.setValue(new Integer(ROT_INIT));
			yField.setValue(new Integer(ROT_INIT));
			zField.setValue(new Integer(ROT_INIT));

			if (selectedObject != null) {
				t = selectedObject.getTransformation().getRotation();
				toXField.setValue(Math.toDegrees(t[0]) % 360);
				toYField.setValue(Math.toDegrees(t[1]) % 360);
				toZField.setValue(Math.toDegrees(t[2]) % 360);
			}

			xSlider.setMinimum(ROT_MIN);
			xSlider.setMaximum(ROT_MAX);			
			ySlider.setMinimum(ROT_MIN);
			ySlider.setMaximum(ROT_MAX);			
			zSlider.setMinimum(ROT_MIN);
			zSlider.setMaximum(ROT_MAX);

			break;
		case SCALE:
			xField.setFormatterFactory(new DefaultFormatterFactory(scaleFormatter));
			yField.setFormatterFactory(new DefaultFormatterFactory(scaleFormatter));
			zField.setFormatterFactory(new DefaultFormatterFactory(scaleFormatter));

			xField.setValue(new Double(SCALE_S_INIT/10));
			yField.setValue(new Double(SCALE_S_INIT/10));
			zField.setValue(new Double(SCALE_S_INIT/10));

			if (selectedObject != null) {
				t = selectedObject.getTransformation().getScale();
				toXField.setValue(t[0]);
				toYField.setValue(t[1]);
				toZField.setValue(t[2]);
			}

			xSlider.setMinimum(SCALE_S_MIN);
			xSlider.setMaximum(SCALE_S_MAX);			
			ySlider.setMinimum(SCALE_S_MIN);
			ySlider.setMaximum(SCALE_S_MAX);			
			zSlider.setMinimum(SCALE_S_MIN);
			zSlider.setMaximum(SCALE_S_MAX);

			break;			
		}

		xField.setColumns(5);
		yField.setColumns(5);
		zField.setColumns(5);		

		toXField.addChangeListener(this);
		toYField.addChangeListener(this);
		toZField.addChangeListener(this);

		((JSpinner.DefaultEditor)toXField.getEditor()).getTextField().setColumns(5);
		((JSpinner.DefaultEditor)toYField.getEditor()).getTextField().setColumns(5);
		((JSpinner.DefaultEditor)toZField.getEditor()).getTextField().setColumns(5);



		//For each field, react when user presses "enter"
		JFormattedTextField[] fields = new JFormattedTextField[] {
				xField, yField, zField};

		//Make a new action for "enter"
		for (JFormattedTextField f : fields) {
			f.getInputMap().put(KeyStroke.getKeyStroke(
					KeyEvent.VK_ENTER, 0), "enter");
			f.getActionMap().put("enter", new FieldAction());
		}		

	}

	/**
	 * Sets up the numberFormatters and sliders for the first time.
	 */
	private void setUpFormattersAndSliders() {
		numberFormat = NumberFormat.getIntegerInstance();

		transFormatter = new NumberFormatter(numberFormat);
		transFormatter.setMinimum(new Integer(TRANS_MIN));
		transFormatter.setMaximum(new Integer(TRANS_MAX));		

		rotationFormatter = new NumberFormatter(numberFormat);
		rotationFormatter.setMinimum(new Integer(ROT_MIN));
		rotationFormatter.setMaximum(new Integer(ROT_MAX));

		scaleFormatter = new NumberFormatter(NumberFormat.getNumberInstance());	
		scaleFormatter.setMinimum(new Double(SCALE_F_MIN));
		scaleFormatter.setMaximum(new Double(SCALE_F_MAX));

		transAndRotateToFormatter = new NumberFormatter(numberFormat);
		transAndRotateToFormatter.setMinimum(new Integer(TO_MIN));
		transAndRotateToFormatter.setMaximum(new Integer(TO_MAX));	

		scaleToFormatter = new NumberFormatter(NumberFormat.getNumberInstance());	
		scaleToFormatter.setMinimum(new Double(SCALE_F_MIN));
		scaleToFormatter.setMaximum(new Double(TO_MAX));

		JPanel sliderPanel = new JPanel();		
		transformationPanel.add(sliderPanel);
		sliderPanel.setBounds(0,30,300,80);

		xSlider = new JSlider();
		xSlider.setBounds(0,20,200,20);
		sliderPanel.add(xSlider);
		xSlider.addChangeListener(this);
		xSlider.addMouseListener(this);

		ySlider = new JSlider();
		ySlider.setBounds(0,40,200,20);
		sliderPanel.add(ySlider);
		ySlider.addChangeListener(this);
		ySlider.addMouseListener(this);

		zSlider = new JSlider();
		zSlider.setBounds(0,60,200,20);
		sliderPanel.add(zSlider);
		zSlider.addChangeListener(this);
		zSlider.addMouseListener(this);

		xField = new JFormattedTextField();
		yField = new JFormattedTextField();
		zField = new JFormattedTextField();

		toXField = new JSpinner();
		toYField = new JSpinner();
		toZField = new JSpinner();		

		//Put in the fields
		JPanel fieldPanel = new JPanel();
		fieldPanel.setBounds(300,30,150,100);
		transformationPanel.add(fieldPanel);	
		fieldPanel.add(xField);
		fieldPanel.add(toXField);
		fieldPanel.add(yField);
		fieldPanel.add(toYField);
		fieldPanel.add(zField);
		fieldPanel.add(toZField);

		//Fix sliders
		setSliders();
	}

	/**
	 * Upon creation/loading of a file, instantiates all of the
	 * selected/current pointers to things belonging to that file's model.
	 *
	 */
	private void initVarOnLoad() {
		currentIDList = loadedModel.getObjectIDList();
		currentObjectList = loadedModel.getObjectList();

		selectModel(loadedModel);

		selectionStack = new CMStack<SelectionType>();
		selectionStack.push(new SelectionType(loadedModel));
	}

	/** 
	 * To be used after selection of a Model or ComplexObject.
	 * Updates current/selected variables.
	 * @param display	Whether or not we should tell selectModel()/CO() to displayAttributes()
	 */
	private void initVarOnSelection(boolean display) {
		Model m;
		ComplexObject co;
		SelectionType top;

		//Top of stack is current selection
		top = selectionStack.peek();

		//Is top object Model or ComplexObject?
		if (top.isModel()) {
			m = top.getModel();
			currentIDList = m.getObjectIDList();
			currentObjectList = m.getObjectList();

			//If it has submodels, put them in the list
			if (m.hasSubModels()) {
				currentSubModelList = m.getSubModelList();
			}			

			//make selected object empty...
			selectedObject = null;            
			selectModel(m, display);

		} else if (top.isComplexObject()){
			co = top.getComplexObject();			
			currentIDList = co.getObjectIDList();
			currentObjectList = co.getObjectList();

			//Select this object
			selectedModel = null;            
			selectObject(co, display);
		}
	}	

	/**
	 * Updates currently selected variables.
	 * Always redraws.
	 */
	private void initVarOnSelection() {
		initVarOnSelection(true);
	}

	/**
	 * Selects an object and updates fields pertaining to the current selection.
	 * @param obj object to be selected
	 * @param	display on or off
	 */
	private void selectObject(ComplexObject obj, boolean display) {
		selectedObject = obj;
		isSelectedModel = false;

		//if GUI exists, display
		if (currentAttrLabel != null && display)
			displayAttributes();		
	}
	/**
	 * Selects an object.
	 * @param obj object to be selected
	 */
	private void selectObject(ComplexObject obj) {
		selectObject(obj, true);
	}

	/**
	 * Selects a model for editing.
	 * @param m
	 * @param display	If just doing this in tree creation, we don't want to bother 
	 * 					with displayAttributes()
	 */
	private void selectModel(Model m, boolean display){
		selectedModel = m;
		isSelectedModel = true;

		if (currentAttrLabel != null && display)
			displayAttributes();
	}

	/**
	 * Overload for selectModel. Displays by default.
	 * @param m
	 */
	private void selectModel(Model m){
		selectModel(m, true);
	}    


	/**
	 * Updates display of attributes of currently selected object.
	 * Updates viewer as well. 
	 */
	private void displayAttributes() {

		//STOP TRANSFORMATION
		disableTransformation = true;

		SelectionType top;
		Transformation trans;
		float[] position, rotation, scale;
		float[] fillColor;		
		Color c;		
		String posformat, rotformat, scaleformat;
		String focusAlias = "No selection", 
		name = "No selection", 
		type = "No selection";

		//If there exists a model/object in focus
		top = selectionStack.peek();
		if (top != null) {
			if (top.isModel()) {
				focusAlias = top.getModel().getAlias();
				focusAlias += " (model)";
			} else {
				focusAlias = top.getComplexObject().getAlias();
				focusAlias += " (object)";
			}
		}

		if(isSelectedModel){
			trans = selectedModel.getTransformation();

			colorField.setText("Color: (N/A)");

			name = selectedModel.getAlias();
			type = selectedModel.getObjectType().toString();  
		}
		else{
			trans = selectedObject.getTransformation();

			//Get color
			fillColor = ((ComplexObject) selectedObject).getFillColor();
			c = new Color(fillColor[0], fillColor[1], fillColor[2], fillColor[3]);
			colorField.setText("Color: ("
					+ c.getRed() + ", "
					+ c.getGreen() + ", "
					+ c.getBlue() + ", "
					+ c.getAlpha() + ")");

			name = selectedObject.getAlias();
			type = selectedObject.getObjectType().toString();   
		}
		position = trans.getTranslation();
		rotation = trans.getRotation();
		scale = trans.getScale();

		posformat = String.format("Position: (%.0f, %.0f, %.0f)",
				position[0], position[1], position[2]);            
		positionField.setText(posformat);

		rotformat = String.format("Rotation: (%.0f, %.0f, %.0f)",
				Math.toDegrees(rotation[0]) % 360, 
				Math.toDegrees(rotation[1]) % 360, 
				Math.toDegrees(rotation[2]) % 360);			
		rotationField.setText(rotformat);

		scaleformat = String.format("Scale: (%.2f, %.2f, %.2f)",
				scale[0], scale[1], scale[2]);   
		scaleField.setText(scaleformat);	

		//Update "transformTo" fields
		float[] t;
		switch(whichTransformation) {
		case TRANS: 
			t = position;
			break;
		case ROTATE:
			t = rotation;
			//less than three!
			for (int i = 0; i < 3; i++)
				t[i] = (float) (Math.toDegrees(t[i]) % 360);
			break;
		case SCALE:
			t = scale;
			break;
		default: 
			t = position;
		}

		toXField.setValue(t[0]);
		toYField.setValue(t[1]);
		toZField.setValue(t[2]);

		selectionField.setText("In focus: " + focusAlias);
		nameField.setText("Name: "+ name);		
		typeField.setText("Type: " + type);		

		//Also, we will show it.
		if (loadedModel.getPriorityGLList() == null)
			previewer.forceModel(loadedModel);
		else{
			previewer.forceModelSameGL(loadedModel);
		}

		//ENABLE TRANSFORMATION
		disableTransformation = false;
	}

	/**
	 * Allows user to add a PrimitiveObject to the current selection's list.
	 * Currently, types are hard-coded. We shall investigate other possibilities....
	 * ...perhaps.
	 */
	private void addPrimitiveObject() {

		//Can't add object to directory!
		SelectionType top = selectionStack.peek();
		if (top.isModel()) {
			if (!top.getModel().getFileName().endsWith(".mdl")) {
				//Directory
				JOptionPane.showMessageDialog(null, "Please first create a submodel.");
				return;
			}
		} else if (top.isComplexObject()) {
			JOptionPane.showMessageDialog(null, "You can only add objects to models.");
			return;
		}

		String typeString, name = "New Object";
		int typeInt = -1;
		boolean chosen = false;
		final int NUM_TYPES = 8;			//CHANGE ME if necessary
		ComplexObject newObject;
		ObjectType type = new ObjectType(loadedModel);
		String choices = "";

		//Would be nice to read in a list of primitives from elsewhere.
		//Oh well - can't be helped now.

		//Choose a type
		choices = "Choose a type of object to add.\n";
		choices = choices.concat("0: Cube" 
				+ "\n1: Sphere"
				+ "\n2: Cylinder"
				+ "\n3: Pyramid"
				+ "\n4: Cone"
				+ "\n5: Roof"
				+ "\n6: Half Cylinder Wall"
				+ "\n7: Thick Half Cylinder Wall");
		while (!chosen) {
			typeString = JOptionPane.showInputDialog(null, choices);

			if (typeString == null) {	//cancelled
				//System.out.println("Not adding anything.");
				JOptionPane.showMessageDialog(null, "Not adding anything.");
				chosen = true;
				return;
			} else {
				typeInt = Integer.parseInt(typeString);
				if (typeInt >= 0 && typeInt < NUM_TYPES) {
					chosen = true;
				}
			}			
		}
		switch(typeInt) {		
		case 0:		//cube
			newObject = type.getObjectOf(ObjType.PrimitiveCube);
			break;
		case 1:		//sphere
			newObject = type.getObjectOf(ObjType.PrimitiveSphere);
			break;
		case 2:		//cylinder
			newObject = type.getObjectOf(ObjType.PrimitiveCylinder);
			break;
		case 3:		//pyramid
			newObject = type.getObjectOf(ObjType.PrimitivePyramid);
			break;
		case 4:		//cone		
			newObject = type.getObjectOf(ObjType.PrimitiveCone);
			break;
		case 5:
			newObject = type.getObjectOf(ObjType.PrimitiveRoof);
			break;
		case 6:	
			newObject = type.getObjectOf(ObjType.PrimitiveHalfCylinderWall);
			break;
		case 7:	
			newObject = type.getObjectOf(ObjType.PrimitiveThickHalfCylinderWall);
			break;
		default:
			newObject = type.getObjectOf(ObjType.PrimitiveCube);
		System.out.println("Why are you here?");
		}

		//Choose name
		name = JOptionPane.showInputDialog(null, "Please enter a name for your shiny new " 
				+ newObject.getObjectType().toString()
				+ "\n(Hit cancel or enter a blank to leave this process.)");
		if (name == null) {
			//Cancelled
			return;
		} else {
			newObject.setAlias(name);
		}				

		//Transform the object by 100 in each direction
		newObject.scaleBy(100, 100, 100);

		//Add to object ID list of selection
		currentIDList.add(newObject);
		currentObjectList.add(newObject);

		JOptionPane.showMessageDialog(null, "Created: " + name + ", " + newObject.getObjectType().toString());
		updateTree(FROM_HERE);
	}

	/**
	 * Adds a ComplexObject to the selected model.
	 * Cannot add complex object to complex object, or to top-level model.
	 */
	private void addComplexObject() {
		//Can only add to Model!
		SelectionType top = selectionStack.peek();
		if (top.isModel()) {
			if (!top.getModel().getFileName().endsWith(".mdl")) {
				//Directory
				JOptionPane.showMessageDialog(null, "Please first create a submodel.");
				return;
			}
		} else if (top.isComplexObject()) {
			JOptionPane.showMessageDialog(null, "You can only add objects to models.");
			return;
		}

		String typeString, name = "New ComplexObject";
		int typeInt = -1;
		boolean chosen = false;
		ArrayList<ComplexObject> availableObjects = loadedModel.getComplexObjectList();
		int numTypes = availableObjects.size();

		ComplexObject newObject;
		String choices = "";

		//Choose a type
		choices = "Choose a type of complex object to add.";
		for (int i = 0; i < numTypes; i++) {
			choices = choices.concat("\n" + i + ": " 
					+ availableObjects.get(i).getAlias());
		}
		while (!chosen) {
			typeString = JOptionPane.showInputDialog(null, choices);

			if (typeString == null) {	//cancelled
				return;
			} else {
				typeInt = Integer.parseInt(typeString);
				if (typeInt >= 0 && typeInt < numTypes) {
					chosen = true;
				}
			}			
		}
		newObject = availableObjects.get(typeInt).copy(loadedModel);

		//Choose name
		name = JOptionPane.showInputDialog(null, "Please enter a name for your shiny new " 
				+ newObject.getObjectType().toString()
				+ "\n(Hit cancel or enter a blank to leave this process and use a default.)");
		if (name == null) {
			newObject.setAlias("Object");
		} else {
			newObject.setAlias(name);
		}	

		//Add to object ID list of selection and update the tree.
		currentIDList.add(newObject);
		currentObjectList.add(newObject);		
		updateTree(FROM_HERE);
		JOptionPane.showMessageDialog(null, "Created: " + name + ", " + newObject.getObjectType().toString());
	}


	/**
	 * Adds a new sub-model to the loaded directory.
	 * Assumes directory at this point -- will squeal if wrong.
	 *
	 */
	private void addSubModel() {
		Model newModel;
		SelectionType s;
		File loaded;
		String alias = "newModel" + System.currentTimeMillis();

		//Get an alias
		alias = JOptionPane.showInputDialog("Please give your new model a name.");
		if (alias == null) {			//Cancelled or bad
			return;
		}
		//Assume OK input.

		//If loaded model is a file, then become unhappy.

		//Must be directory for now -- we can't attach submodels
		//to models yet (in file system).
		loaded = new File(loadedFile);
		if (loaded.isDirectory()) {
			JOptionPane.showMessageDialog(null, "just added " + alias);			//testing
			newModel = new Model(alias, previewer.getGL());

			//Add it to the current model
			loadedModel.getSubModelList().add(newModel);	

			//Update the tree
			updateTree(FROM_HERE);		

			//Select it and update stuff
			s = new SelectionType(newModel);
			selectionStack.push(s);
			initVarOnSelection();

		} else {
			JOptionPane.showMessageDialog(null, "We currently lack this functionality.");
		}
	}


	/**
	 * Saves the model under a new filename.
	 * @return success or failure
	 */
	private boolean saveAs() {
		boolean success = false;
		JFileChooser fileChooser = new JFileChooser(Viewer.getDataPath());
		fileChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);		
		int choiceVal = fileChooser.showOpenDialog(Preditor.this);
		File file;

		if (choiceVal == JFileChooser.APPROVE_OPTION) {	//Success
			file = fileChooser.getSelectedFile();
			loadedFile = file.getPath();				//Store the new filename

			loadedModel.writeModel(loadedFile, true);	//Save away!

		} //Else, failure.

		return success;
	}

	/**
	 * Copies the current item in focus.
	 * @return success of method
	 */
	private boolean copy() {
		boolean success = true;
		//make a copy
		clipboard = selectionStack.peek();	

		/*String alias = null;
		if (clipboard.isModel()) {
			alias = clipboard.getModel().getAlias();
		} else {
			alias = clipboard.getComplexObject().getAlias();
		}
		System.err.println("I just copied " + alias);
		 */

		return success;
	}

	/**
	 * Cuts the current item in focus.
	 * @return success of method
	 */
	private boolean cut() {
		SelectionType top;

		//If we're at the top of the model, we can't cut.
		if (selectionStack.size() == 1) {
			System.err.println("Sorry, you can't cut that.");
			return false;
		}

		clipboard = selectionStack.pop();	//Pop current selection
		initVarOnSelection();				//Reset selection
		top = selectionStack.peek();

		//Remove cut item from its parent
		if (clipboard.isModel()) {
			assert(top.isModel());
			//"What's going on? Model can't be child of object.";

			//Remove model from Submodellist
			assert(currentSubModelList != null);			
			currentSubModelList.remove(clipboard.getModel());	

		} else {		//item is object
			currentObjectList.remove(clipboard.getContents());
			currentIDList.remove(clipboard.getComplexObject().getID());

		}

		//redraw -- need to set GL calculated to false for the parent model
		if (top.isModel()) {
			top.getModel().markForRedraw();
		}

		updateTree(FROM_PARENT);	//current selection is parent of removed item.
		displayAttributes();

		/*
		String alias;
		if (clipboard.isModel()) {
			alias = clipboard.getModel().getAlias();
		} else {
			alias = clipboard.getComplexObject().getAlias();
		}
		System.out.println("I just cut " + alias);
		 */

		return true;
	}

	/**
	 * Pastes the current clipboard item into the current place in focus.
	 * Must be appropriate object and target.
	 * Appropriate pastes:
	 * 	Model into top-level model
	 * 	Complex object into model
	 * 	Complex object into complex object -- actually pastes into the
	 * 		target's parent model
	 * Inappropriate pastes:
	 * 	Model into object
	 * 
	 * @return success of method
	 */
	private boolean paste() {
		boolean success = true;

		//Something must be selected in the tree!
		if (modelTree.getSelectionPath() == null ) {
			JOptionPane.showMessageDialog(null,"Please choose a location into which to paste.");
			return false;
		} else if (clipboard == null) {
			return false;
		}

		SelectionType top = selectionStack.peek();

		//Compatible pastes:				
		//Object into model
		if (clipboard.isComplexObject() && top.isModel()) {
			//Make a new CO identical to it, but with top as the super-model
			ComplexObject co = clipboard.getComplexObject().copy(top.getModel());

			currentObjectList.add(co);
			currentIDList.add(co);
			updateTree(FROM_HERE);
			displayAttributes();
		} 
		//Model into top-level model
		else if (clipboard.isModel() && selectionStack.size() == 1) {
			//Make copy of the model
			Model m = clipboard.getModel().copy();
			currentSubModelList.add(m);
			updateTree(FROM_HERE);
			displayAttributes();

		} else if (clipboard.isComplexObject() && top.isComplexObject()) {
			//If object selected, paste into that object's parent.

			SelectionType t = selectionStack.pop(); //pop off object

			//paste copied object into parent
			SelectionType parentModelType = selectionStack.peek();		
			Model parentModel = parentModelType.getModel();
			ComplexObject co = clipboard.getComplexObject().copy(parentModel);
			parentModel.getObjectList().add(co);
			parentModel.getObjectIDList().add(co);

			selectModel(parentModel);
			updateTree(FROM_PARENT);

			selectionStack.push(t);	//return obj to stack
			selectObject(t.getComplexObject());

			displayAttributes();

		}
		//Incompatible pastes:		
		//Object into object
		//Model into object
		//Model into non-top-level model
		else {
			String clipType = "ComplexObject";
			String topType = "ComplexObject";

			if (top.isModel()) {
				topType = "Model";
			} 
			if (clipboard.isModel()) {
				clipType = "Model";
			}			
			System.err.println("Cannot paste " + clipType + " into " + topType + ".");
			success = false;
		}		

		return success;
	}


	/**
	 * Transforms some transformation some amount in some single dimension, 
	 * using some single type of transformation. (I am intentionally vague.)
	 * @param ot	original transformation
	 * @param type	translation (0), rotation (1), or scale (2)
	 * @param dim	x (0), y (1), or z (2)
	 * @param howMuch	amount of transformation
	 * @param transformBy	transform BY how much (true) or TO some place (false)
	 * @return resulting transformation
	 */
	private Transformation transformSingly(Transformation ot, int type, 
			int dim, float howMuch, boolean transformBy) {

		//If transformation is empty, don't do this
		if (ot == null) {
			return null;
		}

		//make a copy
		Transformation t = new Transformation();
		t.transformTo(ot.getTranslation(), ot.getRotation(), ot.getScale());

		float[] tf = new float[3];				//Transformation

		//set up transformation
		if (type == SCALE) {
			tf = new float[] {1f,1f,1f};
		}		
		if (type == ROTATE) {
			howMuch = (float) Math.toRadians(howMuch);
		}

		tf[dim] = howMuch;

		switch (type) {
		case TRANS: 
			if (transformBy) {
				t.translateBy(tf[0], tf[1], tf[2]);
			} else {
				tf = t.getTranslation();
				tf[dim] = howMuch;
				t.translateTo(tf[0], tf[1], tf[2]);

			}
			break;
		case ROTATE:
			if (transformBy) {
				t.rotateBy(tf[0], tf[1], tf[2]);
			} else {
				tf = t.getRotation();
				tf[dim] = howMuch;
				t.rotateTo(tf[0], tf[1], tf[2]);
			}
			break;
		case SCALE:
			if (transformBy) {
				t.scaleBy(tf[0], tf[1], tf[2]);
			} else {
				tf = t.getScale();
				tf[dim] = howMuch;
				t.scaleTo(tf[0], tf[1], tf[2]);
			}
			break;
		}	

		return t;
	}

	/**
	 * Transforms some object some amount in some single dimension, 
	 * using some single type of transformation. (I am intentionally vague.)
	 * @param obj	object to transform
	 * @param type	translation (0), rotation (1), or scale (2)
	 * @param dim	x (0), y (1), or z (2)
	 * @param howMuch	amount of transformation
	 * @param transformBy	transform BY how much (true) or TO some place (false)
	 */
	private void transformSingly(ComplexObject obj, int type, 
			int dim, float howMuch, boolean transformBy) {
		//If object null, don't do this
		if (obj == null) {
			return;
		}

		Transformation t = transformSingly(obj.getTransformation(), 
				type, dim, howMuch, transformBy);
		obj.setTransform(t);
		displayAttributes();		
	}

	/**
	 * Transforms some model some amount in some single dimension, 
	 * using some single type of transformation. (I am intentionally vague.)
	 * @param mod	model to transform
	 * @param type	translation (0), rotation (1), or scale (2)
	 * @param dim	x (0), y (1), or z (2)
	 * @param howMuch	amount of transformation
	 * @param transformBy	transform BY how much (true) or TO some place (false)
	 */
	private void transformSingly(Model mod, int type, 
			int dim, float howMuch, boolean transformBy) {
		//If model null, don't do this
		if (mod == null) {
			return;
		}

		Transformation t = transformSingly(mod.getTransformation(), 
				type, dim, howMuch, transformBy);
		mod.setTransform(t);
		displayAttributes();		
	}

	/**
	 * Transforms some SelectionType some amount in some single dimension, 
	 * using some single type of transformation. (I am intentionally vague.)
	 * @param st	thing to transform
	 * @param type	translation (0), rotation (1), or scale (2)
	 * @param dim	x (0), y (1), or z (2)
	 * @param howMuch	amount of transformation
	 * @param transformBy	transform BY how much (true) or TO some place (false)
	 */
	private void transformSingly(SelectionType st, int type, 
			int dim, float howMuch, boolean transformBy) {
		//If object null, don't do this
		if (st == null) {
			return;
		}

		Transformation t = transformSingly(st.getTransformation(), 
				type, dim, howMuch, transformBy);
		st.setTransform(t);
		displayAttributes();		
	}


	/**
	 * Changes the alias of the item in focus and updates the Model Tree.
	 */
	private void changeAlias() {
		String input;
		SelectionType top = selectionStack.peek();
		if (top.getAlias() == null) {
			JOptionPane.showMessageDialog(null, "Sorry, I can't do that.");
			return;
		}	

		input = JOptionPane.showInputDialog(null, "Please give a new name for " 
				+ top.getAlias(), top.getAlias());
		if (input != null) {
			top.setAlias(input);

			//recreate tree and display new stuff
			selectionStack.pop();
			updateTree(FROM_PARENT);
			displayAttributes();	
		}
	}

	/**
	 * Changes (or adds) a texture to the selected object.
	 *
	 */	
	private void changeTexture() {
		//Choose a file	
		SelectionType top = selectionStack.peek();

		if(top.isComplexObject()){
			ComplexObject co = top.getComplexObject();

			boolean chosen = false;
			String input;
			Integer face = new Integer(-1);
			Integer choice = new Integer(-1);

			boolean remove = false;		//Set to true if user wants to remove texture
			boolean removeAll = false;
			boolean chooseNumbers = false;

			while(!chosen){
				input = JOptionPane.showInputDialog(null, "Would you like to add a texture(1), " 
						+ "\nremove a texture(2), "
						+ "\nremove all textures(3), "
						+ "\nor set textures to numbered panels(4)?");

				if (input == null) {	//cancelled
					return;
				} 

				else {
					try {
						choice = Integer.parseInt(input);
					} catch (NumberFormatException e){};

					if (choice > 0 && choice < 5) {
						if (choice == 2) {
							remove = true;
						} else if (choice == 3) {
							removeAll = true;
						} else if (choice == 4) {
							chooseNumbers = true;
						}
						chosen = true;
					}
					else
						JOptionPane.showMessageDialog(null,"Enter valid choice #");
				}	
			}
			chosen = false;

			//Remove all textures
			if (removeAll == true) {
				co.clearTexturePath();
				displayAttributes();
				return;
			} 
			//Draw numbers on walls
			if (chooseNumbers == true) {
				co.setTexturesToNumbers();
				displayAttributes();
				return;
			}

			//Continue to choose a face and texture
			if(co.getObjectType().toString().equals((new ObjectType("PrimitiveCube").toString())))
			{
				while(!chosen){

					input = JOptionPane.showInputDialog(null, "Enter # of face (0-5)");

					if (input == null) {	//cancelled
						JOptionPane.showMessageDialog(null, "No texture added");
						return;
					} 

					else {
						face = Integer.parseInt(input);

						if (face >= 0 && face < 6) {
							chosen = true;
						}
						else
							JOptionPane.showMessageDialog(null,"Enter valid face #");
					}	
				}
			}
			else if(co.getObjectType().toString().equals( (new ObjectType("PrimitiveCone")).toString()))
			{
				while(!chosen){

					input = JOptionPane.showInputDialog(null, "Enter '0' for base disk, '1' for cone");
					if (input == null) {	//cancelled
						return;
					} 	
					else {
						face = Integer.parseInt(input);	
						if (face >= 0 && face < 2) {
							chosen = true;
						}
						else
							JOptionPane.showMessageDialog(null,"Enter valid face #");
					}	
				}
			}	
			else if(co.getObjectType().toString().equals( (new ObjectType("PrimitiveCylinder")).toString()))
			{
				while(!chosen){	
					input = JOptionPane.showInputDialog(null, "Enter '0' for top disk, '1' for cylinder, '2' for top disk");
					if (input == null) {	//cancelled
						return;
					} 	
					else {
						face = Integer.parseInt(input);
						if (face >= 0 && face < 3) {
							chosen = true;
						}
						else
							JOptionPane.showMessageDialog(null,"Enter valid face #");
					}	
				}
			}
			else if(co.getObjectType().toString().equals( (new ObjectType("PrimitiveHalfCube")).toString()))
			{
				while(!chosen){
					input = JOptionPane.showInputDialog(null, "Enter face # (0-4)");
					if (input == null) {	//cancelled
						return;
					} 

					else {
						face = Integer.parseInt(input);	
						if (face >= 0 && face < 5) {
							chosen = true;
						}
						else
							JOptionPane.showMessageDialog(null,"Enter valid face #");
					}	
				}
			}
			else if(co.getObjectType().toString().equals( (new ObjectType("PrimitiveRoof")).toString()))
			{
				while(!chosen){	
					input = JOptionPane.showInputDialog(null, "Enter face # (0-4)");	
					if (input == null) {	//cancelled
						return;
					} 	
					else {
						face = Integer.parseInt(input);	
						if (face >= 0 && face < 5) {
							chosen = true;
						}
						else
							JOptionPane.showMessageDialog(null,"Enter valid face #");
					}	
				}
			}
			else if(co.getObjectType().toString().equals( (new ObjectType("PrimitivePyramid")).toString()))
			{
				PrimitiveObject pp = (PrimitiveObject) co;
				int numSides = pp.getverticesLength();
				while(!chosen){
					input = JOptionPane.showInputDialog(null, "Enter face #: (0-"+(numSides-1)+")");
					if (input == null) {	//cancelled
						return;
					} 
					else {
						face = Integer.parseInt(input);

						if (face >= 0 && face < numSides) {
							chosen = true;
						}
						else
							JOptionPane.showMessageDialog(null,"Enter valid face #");
					}	
				}
			}
			else if(co.getObjectType().toString().equals( (new ObjectType("PrimitiveSphere")).toString()))
			{
				face = 0;
				JOptionPane.showMessageDialog(null,"Pick texture for sphere");
			}

			//remove texture
			if (remove) {
				co.setTexturePath(face, "none");	
			} else { //add texture
				File file;
				JFileChooser fileChooser = new JFileChooser(Viewer.getDataPath());		

				fileChooser.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
				int choiceVal = fileChooser.showOpenDialog(Preditor.this);

				if (choiceVal == JFileChooser.APPROVE_OPTION) {	
					file = fileChooser.getSelectedFile();
					if(file.isFile()){
						co.setTexturePath(face, file.getName().toLowerCase());	
						co.updateTexture(face,file.getName().toLowerCase());
					}
				}
			}
		}
		displayAttributes();
	}

	/**
	 *  Changes color of selected object.
	 */
	private void changeColor() {

		if (selectedObject != null) {
			float[] oldColor = selectedObject.getFillColor();
			Color c = JColorChooser.showDialog(
					Preditor.this,
					"Choose a color for " + selectedObject.getAlias(),
					new Color(oldColor[0], oldColor[1], oldColor[2]));

			if (c != null) {

				//Set color of selected object
				float[] components = new float[3];
				c.getColorComponents(components);
				selectedObject.setFillColor(components[0], components[1], components[2], 1);

				displayAttributes();
			} 
		}			
	}

	/**
	 * Parses actions performed on the window.
	 */
	public void actionPerformed(ActionEvent e) {
		AbstractButton source = (AbstractButton) e.getSource();		

		//Menu items
		if (source == openFileMenuItem) {
			loadModel(false);
		} else if (source == newFileMenuItem) {
			createModel(false);
		} else if (source == saveMenuItem) {
			loadedModel.writeModel(loadedFile, true);	//changed from true
			JOptionPane.showMessageDialog(null, loadedFile + " saved.");
		}else if (source == saveAsMenuItem) {
			saveAs();
			JOptionPane.showMessageDialog(null, loadedFile + " saved.");
		} else if (source == exitMenuItem) {
			System.exit(0);
		} else if (source == cutMenuItem) {
			cut();
		} else if (source == copyMenuItem) {
			copy();			
		} else if (source == pasteMenuItem) {
			paste();
		} else if (source == addObjectMenuItem) {
			addPrimitiveObject();			
		} else if (source == addComplexObjectMenuItem) {
			addComplexObject();
		} else if (source == addSubModelMenuItem) {
			addSubModel();
		} else if (source == deleteMenuItem) {
			cut();
		} else if (source == helpMenuItem) {
			JOptionPane.showMessageDialog(null, "Please see the User's Manual for help.");
		} 

		else if (source == colorButton) {
			changeColor();
		} else if (source == aliasButton) {
			changeAlias();
		} else if (source == textureButton) {
			changeTexture();
		}

		//Radio buttons
		else if (source == translationRadioButton) {
			whichTransformation = TRANS;
			disableTransformation = true;
			setSliders();
			disableTransformation = false;
		} else if (source == rotationRadioButton) {
			whichTransformation = ROTATE;
			disableTransformation = true;
			setSliders();
			disableTransformation = false;
		} else if (source == scalingRadioButton) {
			whichTransformation = SCALE;
			disableTransformation = true;
			setSliders();
			disableTransformation = false;
		}		
	}

	/**
	 * Controls change of tree.
	 */
	public void valueChanged(TreeSelectionEvent e) {
		SelectionType newSelection;
		Model m, newModel;
		ComplexObject newCO = null;
		SelectionType s;

		while (selectionStack.size() > 1) {
			selectionStack.pop();
		}

		JTree tree = (JTree) e.getSource();
		if (tree.getSelectionPath() != null) {

			Object[] path = tree.getSelectionPath().getPath();
			if(path.length == 1){
				initVarOnSelection();
			}
			for (int i = 1; i < path.length; i++) {

				s = selectionStack.peek();

				ModelTreeNode node = (ModelTreeNode) ((DefaultMutableTreeNode) path[i]) .getUserObject();

				if (node.isModel()) {	//node is a model
					m = s.getModel();
					ArrayList<Model> subModels = null;
					subModels = m.getSubModelList();			  		

					newModel = subModels.get(node.getNum());
					newSelection = new SelectionType(newModel);
					selectionStack.push(newSelection);
					if (i == path.length - 1) {
						initVarOnSelection();
					} else {
						initVarOnSelection(false);
					}
				} else {					//node is a CO
					if (s.isModel()) {
						//get ID
						newCO = s.getModel().getObjectIDList().get(node.id);
					} else {
						newCO = s.getComplexObject().getObjectIDList().get(node.id);
					}
					newSelection = new SelectionType(newCO);
					selectionStack.push(newSelection);
					if (i == path.length - 1) {
						initVarOnSelection();
					} else {
						initVarOnSelection(false);
					}
				}
			}
		}
	}

	/**
	 * Class to store both the name of the model object/submodel
	 * as well as the number that is used to select it.
	 * For use in the model tree.
	 * @author grossh
	 */
	private class ModelTreeNode {
		private int i;
		private boolean m;
		private String text;
		private int id;		//This object's ID, if it's a ComplexObject

		//For Models
		public ModelTreeNode(int i, boolean m, String text) {
			this.i = i;
			this.m = m;
			this.text = text;
		}

		//For objects
		public ModelTreeNode(int i, boolean m, String text, int id) {
			this.i = i;
			this.m = m;
			this.text = text;
			this.id = id;	
		}

		public int getNum() {
			return i;
		}
		public boolean isModel() {
			return m;
		}
		public String toString() {
			return text;
		}
		public void setText(String s) {
			text = s;
		}
	}


	/**
	 * Updates the tree.
	 * Starts either from the parent of the current selection, or the current selection.
	 * To be used whenever an object is removed or added from the model.
	 * Caveat: populateTree() uses the current selection as the root
	 * point for tree creation. MAKE SURE you have selected the appropriate
	 * thing when you decide whether to use FROM_PARENT or FROM_HERE.
	 * @param startPoint FROM_PARENT, to update the tree from the parent of the current
	 * 					selection, or FROM_HERE, to update the tree starting with the
	 * 					current selection.
	 */
	private void updateTree(int startPoint) {
		TreePath tp = modelTree.getSelectionPath();
		Object[] path = tp.getPath();	
		DefaultMutableTreeNode start;

		//if FROM_PARENT, get parent
		if (startPoint == FROM_PARENT) {
			TreePath pp = tp.getParentPath();
			Object[] ppath = pp.getPath();
			start = (DefaultMutableTreeNode) ppath[ppath.length - 1];
		} 
		//otherwise, FROM_HERE
		else {
			start = (DefaultMutableTreeNode) path[path.length -1];
		}

		start.removeAllChildren(); //reset children
		populateTree(start);

		//remake the tree
		modelTree = new JTree((DefaultMutableTreeNode) path[0]);
		modelTree.addTreeSelectionListener(this);
		treeScrollPane.setViewportView(modelTree);

		if (startPoint == FROM_PARENT) {
			modelTree.expandPath(tp.getParentPath());	
		} else {
			modelTree.expandPath(tp);
		}
	}

	/**
	 * Populates the model tree according to the current list of aliases
	 * and available objects.
	 * Effectively added by Henry Gross, edited by Debbie Chasman.
	 */
	private void populateTree (DefaultMutableTreeNode parent) {
		SelectionType s;			//Current selection
		SelectionType newSelection; //New selection
		int totalObjects = 0;
		int totalModelsAndObjects = 0;
		Model m, newModel;
		ComplexObject co = null, newCO = null;
		ArrayList<ComplexObject> objects = null;
		ArrayList<Model> subModels = null;

		s = selectionStack.peek();    
		if (s.isModel()) {				//s is a model						
			m = s.getModel();
			objects = m.getObjectList();

			for (ComplexObject o : objects) {
				newCO = o;
				DefaultMutableTreeNode node = new DefaultMutableTreeNode(new ModelTreeNode(totalObjects, false, 
						newCO.getAlias(), 
						newCO.getID()));													 
				parent.add(node);        
				newSelection = new SelectionType(newCO);
				selectionStack.push(newSelection);
				initVarOnSelection(false);

				populateTree(node);

				selectionStack.pop();
				initVarOnSelection(false);
				totalObjects++;
			}

			totalModelsAndObjects = totalObjects;

			if (m.hasSubModels()) {
				subModels = m.getSubModelList();

				for (int i = 0; i < subModels.size(); i++) {
					DefaultMutableTreeNode node = new DefaultMutableTreeNode(new ModelTreeNode(totalModelsAndObjects, true, subModels.get(i).getAlias()));
					parent.add(node);

					newModel = subModels.get(i);
					newSelection = new SelectionType(newModel);
					selectionStack.push(newSelection);
					initVarOnSelection(false);

					populateTree(node);

					selectionStack.pop();
					initVarOnSelection(false);
					totalModelsAndObjects++;
				}
			}

		} else {						//s is a complex object

			co = s.getComplexObject();
			objects = co.getObjectList();

			//Go through objects
			if (objects.size() != 0) {
				for (ComplexObject o : objects) {
					newCO = o;

					DefaultMutableTreeNode node = new DefaultMutableTreeNode(new ModelTreeNode(totalObjects, false, 
							newCO.getAlias(), newCO.getID())); 
					parent.add(node);

					if (newCO == null) {
						System.err.println("STOP");
					}
					newSelection = new SelectionType(newCO);
					selectionStack.push(newSelection);
					initVarOnSelection(false);

					populateTree(node);

					selectionStack.pop();
					initVarOnSelection(false);
					totalObjects++;
				}				
			}
			totalModelsAndObjects = totalObjects;
		}		
	}

	/**
	 * Creates the model tree.
	 */
	private void createTree() {
		//After populating the tree, you want reselect what you had selected
		CMStack<SelectionType> tempStack = new CMStack<SelectionType>();

		while (selectionStack.size() > 1) 
			tempStack.push(selectionStack.pop());

		DefaultMutableTreeNode root = new DefaultMutableTreeNode(new 
				ModelTreeNode(0, true, loadedModel.getAlias()));

		selectionStack.push(new SelectionType(loadedModel));
		initVarOnSelection();
		populateTree(root);
		selectionStack.pop();

		modelTree = new JTree(root);
		modelTree.addTreeSelectionListener(this);
		treeScrollPane.setViewportView(modelTree);

		//Reselect everything?
				while (tempStack.size() > 0) {
					selectionStack.push(tempStack.pop());
				}
				initVarOnSelection();		
	}

	/**
	 * Auto-generated method for setting the popup menu for a component
	 */
	private void setComponentPopupMenu(
			final java.awt.Component parent,
			final javax.swing.JPopupMenu menu) {
		parent.addMouseListener(new java.awt.event.MouseAdapter() {
			public void mousePressed(java.awt.event.MouseEvent e) {
				if (e.isPopupTrigger())
					menu.show(parent, e.getX(), e.getY());
			}
			public void mouseReleased(java.awt.event.MouseEvent e) {
				if (e.isPopupTrigger())
					menu.show(parent, e.getX(), e.getY());
			}
		});
	}

	/**
	 * Manages actions performed upon the text fields.
	 * Inspired by Java's SliderDemo3.java's Formatted Text Fields
	 * http://java.sun.com/docs/books/tutorial/uiswing/components/slider.html
	 */
	private class FieldAction extends AbstractAction {

		public void actionPerformed(ActionEvent e) {

			Object source = e.getSource();
			int whichDim = X;
			Number sn;
			float sf = 0;
			JFormattedTextField byField = xField;

			if (source.equals(xField) || source.equals(yField) || source.equals(zField)) {
				//return if transformation currently disabled
				if (disableTransformation) {
					return;
				}				
				byField = (JFormattedTextField) source;	

				//Get proper field and accompanying slider
				if (byField.equals(xField)) {
					whichDim = X;
				} else if (byField.equals(yField)) {
					whichDim = Y;
				} else if (byField.equals(zField)) {
					whichDim = Z;
				}		

				if (!((JFormattedTextField) source).isEditValid()) {
					//Input not valid for formatter in use
					byField.selectAll();
				} else try {
					//Input valid
					byField.commitEdit();

					//Change slider, transform (but disable for now)
					disableTransformation = true;

					if (whichTransformation == SCALE) {
						sn = (Number) byField.getValue();
						sf = sn.floatValue();

					} else {	//translation or rotation
						sf = (Integer) byField.getValue();						
					}	
					transformSingly(selectionStack.peek(), whichTransformation, 
							whichDim, sf,
							true);	

					disableTransformation = false;					

				} catch (ParseException ex) {
					System.err.println("Nooo!");
				}					
			}		
		}	
	}


	/**
	 * Manages changing states of sliders and spinners
	 */
	public void stateChanged(ChangeEvent e) {

		//Ignore any action if transformation is disabled or nothing selected
		if (disableTransformation) {
			return;
		}		
		if(isSelectedModel){
			if(selectedModel == null)
				return;
		}
		else{
			if(selectedObject == null)
				return;
		}

		//Sliders!
		if (e.getSource().getClass() == JSlider.class) {
			JSlider source = (JSlider) e.getSource();
			JFormattedTextField field = xField;		//Field associated with slider
			int whichDim = X;			

			if (source.equals(xSlider)) {
				field = xField;
				whichDim = X;
			} else if (source.equals(ySlider)) {
				field = yField;
				whichDim = Y;
			} else if (source.equals(zSlider)) {
				field = zField;
				whichDim = Z;
			}

			if (!source.getValueIsAdjusting()) {
				//User has removed grip on slider				
				disableTransformation = true;
				//Reset the slider and the field		
				if (whichTransformation == SCALE) {		
					source.setValue(SCALE_S_INIT);
					field.setValue(source.getValue() / 10.);

				} else {	//Translation or rotation both start at 0					
					source.setValue(0);		
					field.setValue(source.getValue());
				}   
				disableTransformation = false;
			} else { 
				//Make the field change with the value
				disableTransformation = true;
				if (whichTransformation == SCALE) {
					field.setValue( ((double) source.getValue()) / 10.);
				} else {	//rotation or scale
					field.setValue(source.getValue());
				}
				disableTransformation = false;

				if (sliding) {
					//transform the original transformation by the current slider values
					Transformation t;
					float sf;
					if (whichTransformation == SCALE) {
						Number sn = (Number) field.getValue();
						sf = sn.floatValue();	
					} else {
						sf = (Integer) field.getValue();
					}
					t = transformSingly(tempTrans, whichTransformation, 
							whichDim, sf, true);

					selectionStack.peek().setTransform(t);
					displayAttributes();
				}
			}
		}
		//Manage Spinner Fields
		else if (e.getSource().getClass() == JSpinner.class) {
			Object source = e.getSource();
			int whichDim = X;
			Number sn;
			float sf = 0;
			JSpinner toField = toXField;

			toField = (JSpinner) source;

			//Get proper fields and accompanying slider
			if (toField.equals(toXField)) {
				whichDim = X;
			} else if (toField.equals(toYField)) {
				whichDim = Y;
			} else if (toField.equals(toZField)) {
				whichDim = Z;
			}				

			try {
				disableTransformation = true;
				if (whichTransformation == SCALE) {
					sn = (Number) toField.getValue();
					sf = sn.floatValue();
				} else {
					sn = (Number) toField.getValue();
					sf = sn.intValue();
				}		
				SelectionType top = selectionStack.peek();				
				transformSingly(top, whichTransformation, 
						whichDim, sf,
						false);	
				disableTransformation = false;
			} catch (Exception ex) {
				System.err.println("An exception occurred.");
				ex.printStackTrace();
			}
		}
	}


	/**
	 * Not currently implemented.
	 */
	public void mouseClicked(MouseEvent e) {

	}

	/**
	 * Not currently implemented.
	 */
	public void mouseEntered(MouseEvent e) {

	}

	/**
	 * Not currently implemented.
	 */
	public void mouseExited(MouseEvent e) {

	}

	/**
	 * Captures object's transformation at the point the slider is first 
	 * grabbed.
	 * @param e	mouse press
	 */
	public void mousePressed(MouseEvent e) {
		Object source = e.getSource();

		if (source.equals(xSlider) || source.equals(ySlider) || source.equals(zSlider)) {
			tempTrans = selectionStack.peek().getTransformation();
			sliding = true;
		}
	}
	/**
	 * Resets that captured transformation when slider is released.
	 * @param e mouse release
	 */
	public void mouseReleased(MouseEvent e) {
		sliding = false;			//No longer using slider
		tempTrans = null;
	}


}
