package model;

/**
 * Keeps track of the transformations made in terms of scale, rotation, 
 * and translation, as well as maintaining a transformation matrix.
 */
public class Transformation {
	private Matrix4x4f matrix;
    private float[] translation, rotation, scale; 
    private boolean isMatrixCurrent;
    
    /**
     * Creates a new transformation with default values.
     */
    public Transformation(){
        reset();
    }
    
    /**
     * Resets the transformation to the default values:
     *      translation = rotation = 0.0 for x,y,z
     *      scale = 1.0 for x,y,z
     *      matrix = identity matrix
     */
    public void reset(){
        translation = new float[3];
        rotation = new float[3];
        scale = new float[3];
        
        scale[0] = scale[1] = scale[2] = 1;
        matrix = new Matrix4x4f();
        isMatrixCurrent = true;
    }
    
    /**
     * If the current transformation matrix isn't up to date, it updates it
     * before returning it. 
     * @return the current transformation matrix
     */
    public Matrix4x4f getTransMatrix(){
        if(!isMatrixCurrent)
            updateMatrix();
        
        return matrix;
    }
    
    /**
     * Returns a deep copy of this transformation object.
     * @return	copy of this transformation object
     */
    public Transformation getTransformation() {
    	Transformation t = new Transformation();
    	t.translation = this.getTranslation();
    	t.rotation = this.getRotation();
    	t.scale = this.getScale();
    	t.matrix = this.getTransMatrix();
    	t.isMatrixCurrent = this.isMatrixCurrent;
    	
    	return t;
    }
    
    /**
     * Returns the translation amounts in a float array.
     * @return translation
     */
    public float[] getTranslation(){
        float[] t = new float[3];
        System.arraycopy(translation, 0, t, 0, 3);
        return t;
    }
    
    /**
     * Returns the rotation amounts in a float array.
     * @return rotation
     */
    public float[] getRotation(){
        float[] r = new float[3];
        System.arraycopy(rotation, 0, r, 0, 3);
        return r;
    }
    
    /**
     * Returns the scale amounts in a float array.
     * @return scale
     */
    public float[] getScale(){
        float[] s = new float[3];
        System.arraycopy(scale, 0, s, 0, 3);
        return s;
    }
    
    /**
     * Invokes a translation on this transformation.
     * @param transX amount in x direction
     * @param transY amount in y direction
     * @param transZ amount in z direction
     */
    public void translateBy(float transX, float transY, float transZ){
        if(transX!=0 || transY!=0 || transZ!=0){
            translation[0] += transX;
            translation[1] += transY;
            translation[2] += transZ;
            isMatrixCurrent = false;
        }
    }
    /**
     * Invokes a rotation on this transformation.
     * @param rotX amount around x axis
     * @param rotY amount around y axis
     * @param rotZ amount around z axis
     */
    public void rotateBy(float rotX, float rotY, float rotZ){
        if(rotX!=0 || rotY!=0 || rotZ!=0){
            rotation[0] += rotX;
            rotation[1] += rotY;
            rotation[2] += rotZ;
            isMatrixCurrent = false;
        }
    }
    /**
     * Invokes a scale on this transformation.
     * @param scaleX amount in x direction
     * @param scaleY amount in y direction
     * @param scaleZ amount in z direction
     */
    public void scaleBy(float scaleX, float scaleY, float scaleZ){
        if(scaleX!=1 || scaleY!=1 || scaleZ!=1){
            scale[0] *= scaleX;
            scale[1] *= scaleY;
            scale[2] *= scaleZ;
            isMatrixCurrent = false;
        }
    }
    /**
     * Sets the translation for this transformation to the specified amount.
     * @param transX amount in x direction
     * @param transY amount in y direction
     * @param transZ amount in z direction
     */
	public void translateTo(float transX, float transY, float transZ){
        if(transX!=0 || transY!=0 || transZ!=0){
            translation[0] = transX;
            translation[1] = transY;
            translation[2] = transZ;
            isMatrixCurrent = false;
        }
    }
    /**
     * Sets the rotation for this transformation to the specified amount.
     * @param rotX amount around x axis
     * @param rotY amount around y axis
     * @param rotZ amount around z axis
     */
    public void rotateTo(float rotX, float rotY, float rotZ){
        if(rotX!=0 || rotY!=0 || rotZ!=0){
            rotation[0] = rotX;
            rotation[1] = rotY;
            rotation[2] = rotZ;
            isMatrixCurrent = false;
        }
    }
    /**
     * Sets the scale for this transformation to the specified amount.
     * @param scaleX amount in x direction
     * @param scaleY amount in y direction
     * @param scaleZ amount in z direction
     */
    public void scaleTo(float scaleX, float scaleY, float scaleZ){
        if(scaleX!=1 || scaleY!=1 || scaleZ!=1){
            scale[0] = scaleX;
            scale[1] = scaleY;
            scale[2] = scaleZ;
            isMatrixCurrent = false;
        }
    }
    /**
     * Translates, rotates, and scales the transformation by the specified amounts.
     * @param t amount to translate in each direction
     * @param r amount to rotate about each axis
     * @param s amount to scale in each direction
     */
    public void transformBy(float[] t, float[] r, float[] s){
        if(t.length == r.length && r.length == s.length && s.length == 3){
            translateBy(t[0], t[1], t[2]);
            rotateBy(r[0], r[1], r[2]);
            scaleBy(s[0], s[1], s[2]);
        }
    }
    /**
     * Sets the transformation's translation, rotation, and scale to the specified amounts.
     * @param t amount to translate in each direction
     * @param r amount to rotate about each axis
     * @param s amount to scale in each direction
     */
	public void transformTo(float[] t, float[] r, float[] s){
        if(t.length == r.length && r.length == s.length && s.length == 3){
            translateTo(t[0], t[1], t[2]);
            rotateTo(r[0], r[1], r[2]);
            scaleTo(s[0], s[1], s[2]);
        }
    }
    
    private void updateMatrix(){
        matrix.setTransformations(translation, rotation, scale);
        isMatrixCurrent = true;
    }
}
