package model;

import javax.media.opengl.GL;

/**
 * Organizes the gl lists used for drawing all the objects in the model. Used to
 * start new gl lists, delete old ones, and call existing ones.
 */
public class PriorityList {
    private static final int DEFAULT_MAX_PRIORITY = 100;
    private static final int DEFAULT_MIN_PRIORITY = 0;
    
    private int firstAssignedListNum;
    private int maxPriority;
    private int minPriority;
    private int baseListNum;
    private GL gl;

    public PriorityList(GL gl){
        this(gl, DEFAULT_MAX_PRIORITY);
    }
    
    public PriorityList(GL gl, int maxPriority){
        if(maxPriority>=0)
            this.maxPriority = maxPriority;
        else
            this.maxPriority = DEFAULT_MAX_PRIORITY;
        
        minPriority = DEFAULT_MIN_PRIORITY;
        this.gl = gl;
        
        firstAssignedListNum = gl.glGenLists(maxPriority - minPriority + 1);
        baseListNum = firstAssignedListNum - minPriority;
    }
    
    /**
     * Returns the maximum usable gl list number
     * @return maximum gl list number
     */
    public int getMaxPriority() {
        return maxPriority;
    }

    /**
     * Returns the minimum usable gl list number
     * @return minimum gl list number
     */
    public int getMinPriority() {
        return minPriority;
    }
    
    /**
     * Returns the actual gl list number for the specified gl list
     * @param priority
     * @return actual gl list number
     */
    public int getGLListNum(int priority){
        if(0 <= priority && priority <= maxPriority && baseListNum >= 0){
            return baseListNum + priority;
        }
        return -1;
    }
    
    /**
     * Create the specified gl list
     * @param priority gl list to create
     */
    public boolean plCreateList(int priority){
        if(0 > priority || priority > maxPriority || baseListNum < 0)
            return false;
        gl.glDeleteLists(baseListNum + priority, 1);
        gl.glNewList(baseListNum + priority, GL.GL_COMPILE);
        return true;
    }
    
    /**
     * Call the specified gl list
     * @param priority gl list to call
     */
    public void plCallList(int priority){
    	if(0 <= priority && priority <= maxPriority && baseListNum >= 0)
            gl.glCallList(baseListNum + priority);
    }
    
    /**
     * Delete the specified gl list
     * @param priority gl list to delete
     */
    public void plDeleteList(int priority){
        if(0 <= priority && priority <= maxPriority)
            gl.glDeleteLists(baseListNum + priority, 1);
    }
    
    public void plDeleteAllLists(){
        gl.glDeleteLists(baseListNum, maxPriority - minPriority + 1);
    }
}
