package model;

/**
 * Stores the type of a particular 3d object. Used in ComplexObject to tell accessors
 * what type of object it is. Used in Model to instantiate new objects of a given type.
 */
public class ObjectType {
    /**
     * These are all the ObjectTypes supported by the program.
     * To add a new ObjectType, add it to the list and then to the getStringOf(...)
     * and getObjectOf(...) methods below.
     */
    public static enum ObjType { Undefined, Model, PrimitiveCube, PrimitiveSphere,
                                 PrimitiveCylinder, PrimitivePyramid, PrimitiveCone, 
                                 PrimitiveHalfCube, PrimitiveRoof, PrimitiveHalfCylinderWall,
                                 PrimitiveThickHalfCylinderWall, Complex }
    
    protected ObjType type;
    protected Model model;
    protected String complexType = "";
    
    public ObjectType(){
        type = ObjType.Undefined;
    }
    public ObjectType(ObjType type){
        this.type = type;
    }
    public ObjectType(String type){
        setObjType(type);
    }
    public ObjectType(Model model){
        this();
        this.model = model;
    }
    public ObjectType(Model model, ObjType type){
        this(type);
        this.model = model;
    }
    public ObjectType(Model model, String type){
        this(type);
        this.model = model;
    }
    
    public ObjectType copy(Model m){
        ObjectType ot = new ObjectType(m);
        ot.type = type;
        ot.complexType = complexType;
        
        return ot;
    }
    
    public void setObjType(ObjType type){
        this.type = type;
    }
    
    public void setObjType(String type){
        this.type = getObjTypeOf(type);
        if(this.type == ObjType.Complex && type.length() > 8){
            complexType = type.substring(8);
        }
    }
    
    public ObjType getObjType(){
        return type;
    }
    
    public String getComplexType(){
        return complexType;
    }
    
    public void setComplexType(String t){
    	complexType = t;
    }
    
    public ComplexObject getObject(){
        return getObjectOf(type);
    }
    
    public String toString(){
        return getStringOf(type);
    }
    
    /**
     * Returns the String representation of a given type.
     * When adding a new ObjectType, see the example given within the method below.
     * @param type the ObjectType whose String representation is desired
     * @return the String representation of the type
     */
    public static String getStringOf(ObjType type){
        switch(type){
            case PrimitiveCube:
                return "PrimitiveCube";
            case PrimitiveSphere:
                return "PrimitiveSphere";
            case PrimitiveCylinder:
                return "PrimitiveCylinder";
            case PrimitivePyramid:
                return "PrimitivePyramid";
            case PrimitiveCone:
                return "PrimitiveCone";
            case PrimitiveHalfCube:
            	return "PrimitiveHalfCube";
            case PrimitiveRoof:
                return "PrimitiveRoof";
            case PrimitiveHalfCylinderWall:
                return "PrimitiveHalfCylinderWall";
            case PrimitiveThickHalfCylinderWall:
                return "PrimitiveThickHalfCylinderWall";
            
             /* Add statements like the following for your new PrimitiveObject.
             * It is recommended that the naming conventions are maintained.
            
            case PrimitiveSpecialObject:
                return "PrimitiveSpecialObject";
             */
            
            case Complex:
                return "Complex";
            case Model:
                return "Model";
            case Undefined:
            default:
                return "Undefined";
        }
    }
    
    /**
     * Returns a new object of the given type.
     * When adding a new ObjectType, see the example given within the method below.
     * @param type the type of object of which a new instance is desired
     * @return a new instantiation of the given object type
     */
    public ComplexObject getObjectOf(ObjType type){
        switch(type){
            case PrimitiveCube:
                return new PrimitiveCube(model);
            case PrimitiveSphere:
                return new PrimitiveSphere(model);
            case PrimitiveCylinder:
                return new PrimitiveCylinder(model);
            case PrimitivePyramid:
                return new PrimitivePyramid(model);
            case PrimitiveCone:
                return new PrimitiveCone(model);
            case PrimitiveHalfCube:
            	return new PrimitiveHalfCube(model);
            case PrimitiveRoof:
                return new PrimitiveRoof(model);
            case PrimitiveHalfCylinderWall:
                return new PrimitiveHalfCylinderWall(model);
            case PrimitiveThickHalfCylinderWall:
                return new PrimitiveThickHalfCylinderWall(model);
            
            /* Add statements like the following for your new PrimitiveObject.
            * It is recommended that the naming conventions are maintained.
           
           case PrimitiveSpecialObject:
               return new PrimitiveSpecialObject(model);
            */
            
            case Complex:
                return new ComplexObject(model);
            case Undefined:
            default:
                return null;
        }
    }
    
    public static ObjType getObjTypeOf(String type){
        if(type.startsWith("Complex-")){
            return ObjType.Complex;
        }
        for(ObjType t : ObjType.values()){
            if(type.equalsIgnoreCase(getStringOf(t)))
                return t;
        }
        
        return ObjType.Undefined;
    }
}
