package model;

import model.ComplexObject;

import java.io.*;
import java.util.*;
import javax.media.opengl.*;
import javax.media.opengl.glu.*;
import com.sun.opengl.util.*;

/**
 * The overarching class that stores the entire model in memory. Also used for
 * submodels. Containes methods for accessing submodels and objects, and file and
 * graphical I/O.
 */
public class Model {
    public static final int NUM_PRIORITY_GL_LISTS = 5;
    
    protected String alias;				//Added by Debbie
    protected String fileName;
    protected ArrayList<Model> subModels;
    protected ArrayList<ComplexObject> objects;
    protected ArrayList<ComplexObject> complexObjects;
    protected PriorityList priorityGLList;
    protected IDList objectIDList;
    protected ObjectType objectType;
    protected Transformation transformation;
    protected GL gl;
    protected GLU glu;
    protected GLUT glut;
    protected boolean hasSubModels;
    protected boolean glCalculated;
    
    public Model(){
        resetModel();
    }
    public Model(File path){
        this();
        readModel(this, path);
    }
    public Model(GL gl, GLU glu, GLUT glut){
        this();
        setGL(gl);
        setGLU(glu);
        setGLUT(glut);
    }
    public Model(Model parentModel, File path, GL gl, GLU glu, GLUT glut){
        this(gl, glu, glut);
        readModel(parentModel, path);
    }
    
    /**
     * Constructs a new Model object that does not already have a file
     * associated with it.
     * @param alias alias of new model (Example: "TopSecretHideout")
     * @param g GL belonging to viewer currently in use (necessary for 
     * 			proper assembly of priority lists)
     */
    public Model(String alias, GL g) {
    	this();
    	fileName = alias + ".mdl";
    	setAlias(alias);
    	gl = g;
    }
    
    /**
     * Makes a deep copy of this model.
     * @return	copy of the model
     */
    public Model copy() {
    	ComplexObject temp;
    	
    	Model newM = new Model(this.gl, this.glu, this.glut);
    	
    	//copy submodels
        newM.subModels = new ArrayList<Model>();
        for (Model sub : this.subModels) {
        	newM.subModels.add(sub.copy());
        }

        //copy objects
        newM.objects = new ArrayList<ComplexObject>();        
        newM.objectIDList = new IDList();
        for (ComplexObject c : this.objects) {
        	temp = c.copy(newM);
        	newM.objects.add(temp);
        	newM.objectIDList.add(temp);
        }
        
        //set up other stuff
        newM.setTransformation(this.transformation.getTransformation());
        newM.hasSubModels = this.hasSubModels();
        newM.setAlias("Copy of " + alias);
    	
    	return newM;
    }
    
    /**
     * Resets the model. For use in constructor and in loading a new
     * model file into this model.
     */
    private void resetModel() {
        subModels = new ArrayList<Model>();
        objects = new ArrayList<ComplexObject>();
        complexObjects = new ArrayList<ComplexObject>();
        if(priorityGLList != null)
            priorityGLList.plDeleteAllLists();
        priorityGLList = null;
        objectIDList = new IDList();
        objectType = new ObjectType("Model");
        transformation = new Transformation();
        hasSubModels = false;
        setAlias("Nameless Model");
    }
    
    /**
     * Marks this model for redraw.
     */
    public void markForRedraw() {
    	glCalculated = false;
    }
    
    /**
     * Gets the filename of this Model.
     * @return name of model's file
     */
    public String getFileName() {
    	return fileName;
    }
    
    /**
     * Added by Debbie
     * Gets the alias belonging to this Model.
     * @return alias
     */
    public String getAlias(){
        return alias;
    }
    /**
     * Added by Debbie.
     * Sets the alias for this Model.
     * @param alias
     */
    public void setAlias(String alias){
        this.alias = alias;
    }
    
    /**
     * Retrieves an object from this model's list by its alias
     * @return ComplexObject
     */
    public ComplexObject getObjByAlias(String objAlias) {
    	ComplexObject co = null;
    	
    	for (ComplexObject c : objects) {
    		if (c.getAlias().equals(objAlias)) {
    			return co;
    		}
    	}   	
    	
    	return co;
    }
    
    public IDList getObjectIDList() {
        return objectIDList;
    }
    public void setObjectIDList(IDList objectIDList) {
        this.objectIDList = objectIDList;
    }
    
    public ObjectType getObjectType(){
        return objectType;
    }
    
    public Transformation getTransformation(){
        return transformation;
    }
    public void setTransformation(Transformation t){
        transformation = t;
    }
    
    public GL getGL(){
        return gl;
    }
    public void setGL(GL gl){
        this.gl = gl;
        priorityGLList = null;
        
        if(hasSubModels){
            for(Model m : subModels)
                m.setGL(gl);
        }
    }
    public GLU getGLU(){
        return glu;
    }
    public void setGLU(GLU glu){
        this.glu = glu;
        
        if(hasSubModels){
            for(Model m : subModels)
                m.setGLU(glu);
        }
    }
    public GLUT getGLUT(){
        return glut;
    }
    public void setGLUT(GLUT glut){
        this.glut = glut;
        
        if(hasSubModels){
            for(Model m : subModels)
                m.setGLUT(glut);
        }
    }
    
    /**
     * Added by Debbie
     * Accesses hasSubModels boolean
     * @return True if has submodels, false otherwise
     */
    public boolean hasSubModels() {
    	return hasSubModels;
    }
    
    /**
     * Sets this model's list of sub-models
     * 
     */
    public void setSubModelList(ArrayList<Model> sml) {
    	subModels = sml;
    	hasSubModels = true;
    }    
    /**
     * Added by Debbie.
     * Returns list of sub-models.
     * @return sub models belonging to this model
     */
    public ArrayList<Model> getSubModelList() {
    	return subModels;
    }
    
    public PriorityList getPriorityGLList() {
        return priorityGLList;
    }
    public void setPriorityGLList(PriorityList priorityGLList) {
        this.priorityGLList = priorityGLList;
    }
    
    public ArrayList<ComplexObject> getObjectList(){
        return objects;
    }
    
    /**
     * Accesses complex objects.
     * @return	list of complex objects in this model
     */
    public ArrayList<ComplexObject> getComplexObjectList() {
    	return complexObjects;
    }
    public void setObjectList(ArrayList<ComplexObject> list){
        objects = list;
    }
    
    public void setTransform(Transformation t) {
        glCalculated = false;
        transformation = t;
    }
    
    public void drawModel(){
        drawModel(false);
    }
    public void drawModel(boolean forceRender){
        checkPriorityListInstantiation();
        drawModel(forceRender, priorityGLList.getMaxPriority());
    }
    public void drawModel(boolean forceRender, int maxPriority){
        checkPriorityListInstantiation();
        drawModel(forceRender, priorityGLList.getMinPriority(), maxPriority);
    }
    public void drawModel(boolean forceRender, int minPriority, int maxPriority){
        // if there are submodels, call drawModel on each of them
        if(hasSubModels){
            for(Model m : subModels)
                m.drawModel(forceRender, minPriority, maxPriority);
        }
        // otherwise just draw the objects in this model
        else{
            checkPriorityListInstantiation();
            
            boolean[] glCalc = new boolean[maxPriority-minPriority+1];
            
            for(int i=minPriority; i<=maxPriority; i++){
                glCalc[i-minPriority] = true;
                for(ComplexObject co : objects){
                    if(!co.getGlCalculated())
                        glCalc[i-minPriority] = false;
                }
            }
            
            for(int i=minPriority; i<=maxPriority; i++){
                if(forceRender || !glCalculated || !glCalc[i-minPriority]){
					priorityGLList.plCreateList(i);
                    gl.glPushMatrix();
                    gl.glMultMatrixf(transformation.getTransMatrix().getMatrix(), 0);
                    for(ComplexObject co : objects){
                        if(co.getPriority() == i)
                            co.draw(true);
                    }
                    gl.glPopMatrix();
                    gl.glEndList();
                }
                priorityGLList.plCallList(i);
            }
        }
        glCalculated = true;
    }
    
    private void checkPriorityListInstantiation(){
        if(priorityGLList == null)
        	priorityGLList = new PriorityList(gl, NUM_PRIORITY_GL_LISTS - 1);
    }
    
    /**
     * Reads in a model.
     * @param parentModel
     * @param file
     */
    public void readModel(Model parentModel, File file){
        File[] files = new File[0];
        File coDir;
        
        if(!file.exists()){
        	System.err.println("Error: model not found.");
        	return;
        }
        
        resetModel();
        
        if(file.isFile()){
        	files = new File[1];
        	files[0] = file;
            fileName = file.getName();
        }
        else if(file.isDirectory()){
            files = file.listFiles();
            fileName = file.getPath();
            alias = file.getName();
        	hasSubModels = true;
        }
        else{
        	System.err.println("Error: model file/directory does not exist.");
        	return;
        }
        
        // if it's a directory, we'll read in the submodels
        if(hasSubModels){
        	Model subModel;
        	String compType;
            
            // first we need to check for a ComplexObjects subdirectory (".co")
            coDir = new File(file.getPath()+"/.co");
            if(coDir.exists() && coDir.isDirectory()){
                ComplexObject co;
                for(File f : coDir.listFiles()){
                    if(f.isFile() && f.getName().endsWith(".mdl")){
                        co = new ComplexObject(this);
                        //Fixed(?) by Debbie -- should not be ALIAS, but Complex Type!
                        compType = readSingleModelFile(parentModel, f, co.getObjectList(), 
                                new IDList(), co.getTransformation(), true, co);
                        co.setAlias(compType);
                        co.setComplexType(compType);                        
                        complexObjects.add(co);
                    }
                }
            }
            
            // now read in all the model files in this model directory
        	for(int i=0; i<files.length; i++){
        		if((files[i].isFile() && files[i].getName().endsWith(".mdl")) ||
                        files[i].isDirectory() && 
                        !files[i].getName().substring(0,1).equals(".")){
                    
                    subModel = new Model(parentModel, files[i], gl, glu, glut);
	        		subModels.add(subModel);
        		}
        	}
        }
        // if it's not a directory, let's just read in this single file
        else{
            transformation = new Transformation();
            setAlias(readSingleModelFile(parentModel, file, objects, objectIDList, 
                    transformation, false, null));
        }
    }
    
    public String readSingleModelFile(Model parentModel, File file, 
            ArrayList<ComplexObject> objList, IDList objIDList, 
            Transformation trans, boolean isCO, ComplexObject co){
        Scanner scanner;
        
        try{
            scanner = new Scanner(file);
        }
        catch(FileNotFoundException e){
            System.err.println("Error: model file not found!");
            return "";
        }
        
        ModelFile modelFile = new ModelFile(scanner);
        //String lineIn;
        ComplexObject obj = null;
        ObjectType type = new ObjectType(this);
        String objTypeStr;
        
        //Read in alias, transformation
        //Added by Debbie, 1-31-07
        String a = modelFile.readString();
        
        if(isCO && co!=null){
            co.setOverwriteChildColors(!(modelFile.readInt()==0));
        }
        
        trans.transformTo(modelFile.readNFloats(3),
                          modelFile.readNFloats(3),
                          modelFile.readNFloats(3));          
        
        //Read in the complex objects
        while(!modelFile.isEndOfFile()){
            obj = null;
            
            //necessary to allow multiple types of complex objects in this model
            type = new ObjectType(this);
            objTypeStr = modelFile.readObjectTypeString();
            type.setObjType(objTypeStr);
            if(type.getObjType() == ObjectType.ObjType.Complex){
                for(ComplexObject c : parentModel.complexObjects){
                    if(c.getAlias().equalsIgnoreCase(type.getComplexType())){
                        obj = c.copy(this);
                        obj.setObjectType(type);
                        break;
                    }
                }
            }
            else{
                obj = type.getObject();
            }
            
            if(!(obj == null)){
                obj.load(modelFile);
                objList.add(obj);
                objIDList.add(obj);
            }
            
            while(!modelFile.isEndOfObject()){
                modelFile.readString();
            }
        }
        
        return a;
    }
    
    /**
     * 
     * @param path - the path to the model directory/file
     * @param overwrite - by default, files will not be overwritten
     */
    public void writeModel(String path, boolean overwrite){
        PrintWriter writer;
        File file = new File(path);
        ModelFile mf = new ModelFile();
        
        // if this model has submodels
        if(hasSubModels){
            if(!file.exists())
                file.mkdirs();
            String pathName = file.getPath();
            
            for(Model subModel : subModels)
                subModel.writeModel(pathName, overwrite);
        }
        // if it's a single model
        else{
            // if it was a submodel, we use the stored file name, or alias if we have to
            if(file.isDirectory()){
                if(fileName != null)
                    file = new File(file.getPath() + "/" + fileName);
                else
                    file = new File(file.getPath() + "/" + alias + ".mdl");
            }
            
            try{
                if(!file.exists())
                    file.createNewFile();
                else if(!overwrite)
                    throw new Exception("Model file already exists and overwrite=false.");
                else{
                    file.delete();
                    file.createNewFile();
                }
                
                writer = new PrintWriter(file);
            }
            catch(Exception e){
                System.err.println("Error: model file \""+path+"\" cannot be saved!");
                return;
            }
            
            //Write out alias/transformation for entire model
            //Added 1-31-07 by Debbie
            mf.writeString(writer, alias);
            mf.writeFloatArray(writer, transformation.getTranslation());	
            mf.writeFloatArray(writer, transformation.getRotation());
            mf.writeFloatArray(writer, transformation.getScale());
            
            //Now we write out each object
            for(ComplexObject co : objects){
                writer.print("ObjectType: ");
                if(co.getObjectType().getObjType() == ObjectType.ObjType.Complex){
                	writer.println("Complex-"+co.getObjectType().getComplexType());
                }
                else{
                    writer.println(co.getObjectType());
                }
                
                co.write(writer, mf);
                
                writer.println("EndObject");
            }
            
            writer.close();
        }
    }
    
    public void texturesOn(boolean b){
        for(Model mo: subModels)
			mo.texturesOn(b);
        for(ComplexObject co: objects)
            co.setTexturesOn(b);
    }
    public void brightnessOn(boolean b){
        for(Model mo: subModels)
			mo.glCalculated = false;
        for(ComplexObject co: objects)
            co.glCalculated = false;
    }
    
}
