package model;

/**
 * Matrix4x4f is a 4x4 matrix of floats.
 * It's stored in a 1-d float array, so it's important to keep in mind
 * that everything is done in column-major order for OpenGL.
 */
public class Matrix4x4f {
    // the identity matrix looks the same in row- and column-major order
    public static final float[] IDENTITY_MATRIX =
            { 1, 0, 0, 0,
              0, 1, 0, 0,
              0, 0, 1, 0,
              0, 0, 0, 1 };
    
    private float[] matrix;
    /*
    private float tX, tY, tZ;
    private float rX, rY, rZ;
    private float sX, sY, sZ;
    */
    
    public Matrix4x4f(){
        this(IDENTITY_MATRIX);
    }
    
    public Matrix4x4f(float[] m){
        matrix = new float[16];
        setMatrix(m);
    }
    
    public Matrix4x4f(Matrix4x4f m){
    	matrix = new float[16];
        setMatrix(m.getMatrix());
    }
    
    public float[] getMatrix(){
        float[] m = new float[16];
        
        for(int i=0; i<16; i++)
            m[i] = matrix[i];
        
        return m;
    }
    
    public float[] getIdentity(){
        float[] m = new float[16];
        
        for(int i=0; i<16; i++)
            m[i] = IDENTITY_MATRIX[i];
        
        return m;
    }
    
    public void setMatrix(Matrix4x4f m){
        setMatrix(m.getMatrix());
    }
    
    public void setMatrix(float[] m){
        if(m.length != 16) return;
        
        for(int i=0; i<16; i++)
            matrix[i] = m[i];
    }
    
    public void swapMajorOrder(){
        float[] m = new float[16];
        
        for(int i=0; i<4; i++)
            for(int j=0; j<4; j++)
                m[i*4+j] = matrix[i+4*j];
        
        matrix = m;
    }
    
    public void multMatrix(Matrix4x4f m){
        multMatrix(m.getMatrix());
    }
    
    public void multMatrix(float[] m){
        if(m.length != 16) return;
        
        float[] t = new float[16];
        
        for(int i=0; i<4; i++)  // column index
            for(int j=0; j<4; j++)  // row index
                for(int k=0; k<4; k++)  // vector index
                    t[i*4+j] += matrix[i*4+k] * m[k*4+j];
        
        matrix = t;
    }
    
    public void multVertices(float[][] original, float[][] result){
        // the second dimension for the float arrays must be at least 4
        // - we're dealing with 3d vertices being multiplied by a matrix
        for(int i=0; i<original.length; i++){
            for(int j=0; j<4; j++){
                result[i][j] = 0;
                for(int k=0; k<4; k++){
                    result[i][j] += original[i][k] * matrix[k*4+j];
                }
            }
        }
    }
    
    /**
     * Applies a translation to the matrix. 
     * @param transX translation in x direction
     * @param transY translation in y direction
     * @param transZ translation in z direction
     */
    public void translateBy(float transX, float transY, float transZ){ 
    	multMatrix(new float[] { 1, 0, 0, 0,
    							 0, 1, 0, 0,
    							 0, 0, 1, 0,
    							 transX, transY, transZ, 1 });
    	
    }
    public void scaleBy(float scaleX, float scaleY, float scaleZ){
        multMatrix(new float[]
                { scaleX, 0, 0, 0,
                  0, scaleY, 0, 0,
                  0, 0, scaleZ, 0,
                  0, 0, 0, 1 });
    }
    public void rotateBy(float rotX, float rotY, float rotZ){
        multMatrix(new float[]
                { (float)(Math.cos(rotY)*Math.cos(rotZ)),
                      (float)(Math.sin(rotX)*Math.sin(rotY)*Math.cos(rotZ) +
                          Math.cos(rotX)*Math.sin(rotZ)), 
                      (float)(-Math.cos(rotX)*Math.sin(rotY)*Math.cos(rotZ) +
                              Math.sin(rotX)*Math.sin(rotZ)), 
                      0,
                  (float)(-Math.cos(rotY)*Math.sin(rotZ)),
                      (float)(-Math.sin(rotX)*Math.sin(rotY)*Math.sin(rotZ) +
                          Math.cos(rotX)*Math.cos(rotZ)), 
                      (float)(Math.cos(rotX)*Math.sin(rotY)*Math.sin(rotZ) +
                              Math.sin(rotX)*Math.cos(rotZ)), 
                      0,
                      (float)(Math.sin(rotY)),
                      (float)(-Math.sin(rotX)*Math.cos(rotY)), 
                      (float)(Math.cos(rotX)*Math.cos(rotY)), 
                      0,
                  0, 0, 0, 1 });
    }
    public void setTransformations(float[] trans, float[] rot, float[] scale){
        float[] m = new float[16];
        
        m[0] = (float)(scale[0]*(Math.cos(rot[1])*Math.cos(rot[2])));
        m[1] = (float)(scale[0]*(Math.sin(rot[0])*Math.sin(rot[1])*Math.cos(rot[2]) + 
               Math.cos(rot[0])*Math.sin(rot[2])));
        m[2] = (float)(scale[0]*(-Math.cos(rot[0])*Math.sin(rot[1])*Math.cos(rot[2]) +
                Math.sin(rot[0])*Math.sin(rot[2])));
        m[3] = 0;
        m[4] = (float)(scale[1]*(-Math.cos(rot[1])*Math.sin(rot[2])));
        m[5] = (float)(scale[1]*(-Math.sin(rot[0])*Math.sin(rot[1])*Math.sin(rot[2]) +
                Math.cos(rot[0])*Math.cos(rot[2])));
        m[6] = (float)(scale[1]*(Math.cos(rot[0])*Math.sin(rot[1])*Math.sin(rot[2]) +
                Math.sin(rot[0])*Math.cos(rot[2])));
        m[7] = 0;
        m[8] = (float)(scale[2]*(Math.sin(rot[1])));
        m[9] = (float)(scale[2]*(-Math.sin(rot[0])*Math.cos(rot[1])));
        m[10] = (float)(scale[2]*(Math.cos(rot[0])*Math.cos(rot[1])));
        m[11] = 0;
        m[12] = trans[0];
        m[13] = trans[1];
        m[14] = trans[2];
        m[15] = 1;
        
        setMatrix(m);
    }
}
