//#include <mysql/mysql.h>
#include <stdlib.h>
#include <iostream>

using namespace std;

#include "Db_Query.h"

//Constructor: Opens a new connection to the database.
Db_Query::Db_Query()
{
	//initialize storedResults to nulls
	for(int index = 0; index < kMaxResults; index++)
		storedResults[index] = NULL;
	numResults = 0;

	open();
	error = NULL;
}


//Opens a connection to the database
void
Db_Query::open()
{
	conn = mysql_init(NULL);
	if(conn == NULL)
	{
		valid = false;
		return;
	}

	conn = mysql_real_connect(conn, host, username, 
		password, database, 0, NULL, 0);
	
	if(conn == NULL)
	{
		valid = false;
		return;
	}
	else
		valid = true;
}	

//checks if this object has a valid database connection
bool
Db_Query::isValid()
{
	return valid;
}

//Destructor: cleans up by taking the following actions:
// -closes the open MYSQL connection
// -releases all store data
Db_Query::~Db_Query()
{
	close();

	for(int count=0; count < kMaxResults; count++)
		release(storedResults[count]);

	numResults = 0;
}

//Closes the open MYSQL connection
void
Db_Query::close()
{
	mysql_close(conn);
	valid = false;
}

//Submits a query to the database, and ignores the result
//Returns 1 on success, 0 on error.
int
Db_Query::query(char *query, int len)
{
#ifdef DB_Q_DEBUG
        cout << "Query: \n" << query << endl;
#endif
	if( 0 != mysql_real_query(conn, query, len))
	{
		error = mysql_error(conn);
		return 0;
	}
	else return 1;
}

//Submits a query to the database
const sql_data_t
Db_Query::query(char * query, int len, int *nrows, int *nfields)
{
	//check for an open location to put the query in
	if(numResults >= kMaxResults)
	{
		error = "Maximum result sets already stored.";
		return NULL;
	}


	MYSQL_RES * res_set;
	MYSQL_ROW row;
	char * fieldStore;
	int resultIndex = 0;

#ifdef DB_Q_DEBUG
	cout << "Query: \n" << query << endl;
#endif

	// execute the query
	if( 0 != mysql_real_query(conn, query, len))
	{
		error = mysql_error(conn);
		return NULL;
	}

	res_set = mysql_store_result(conn);

	/* empty result set */
	if(res_set == NULL)
	{
		if(nrows != NULL)
			*nrows = 0;
		if(nfields != NULL)
			*nfields = 0;

		sql_data_t results = new char *[1];
		*results = new char[1] ;
		**results = '\0';
		error = kSuccess;
		return (const sql_data_t) results;
	}

	// if it's not an empty result set...
	int num_rows = mysql_num_rows(res_set);
	int num_fields = mysql_num_fields(res_set);

        sql_data_t results = new char *[num_rows * num_fields];
	for(int count=0; count < num_rows * num_fields; count++) results[count] = NULL;

	// check to make sure we've got space for another set of results
	while(resultIndex < kMaxResults && storedResults[resultIndex] != NULL)
		resultIndex++;
	if(resultIndex >= kMaxResults)
	{
		error = "Error: No room in result set storage.  NumResults and StoredResults inconsistant!";
		return NULL;
	}
	storedResults[resultIndex] = results;
	resultLengths[resultIndex] = num_rows * num_fields;
	
	if(!results)
	{
		error = "Out of memory.";
		return NULL;
	}

	// organize the query results into an array
	for(int rowCnt=0; rowCnt < num_rows; rowCnt++)
	{
		row = mysql_fetch_row(res_set);
		if(row == NULL)
		{
			release(results);
			return NULL;
		}
		for(int fieldCnt = 0; fieldCnt < num_fields; fieldCnt++)
		{

			if(row[fieldCnt])
			{
				fieldStore = new char[strlen(row[fieldCnt]) + 1];
				results[rowCnt * num_fields + fieldCnt] 
					= fieldStore; 
				strcpy(fieldStore,
					row[fieldCnt]);
			}
			else //null field contents
			{
				fieldStore = new char[strlen(kNULLRep) + 1];
				results[rowCnt * num_fields + fieldCnt] 
					= fieldStore; 
				strcpy(fieldStore, kNULLRep);
			}
		}
	}

	mysql_free_result(res_set);

	error = kSuccess;
	if(nrows != NULL)
		*nrows = num_rows;
	if(nfields != NULL)
		*nfields = num_fields;
	return (const sql_data_t)results;
}

//Returns the text of the last error to occur
const char *
Db_Query::getError()
{
	return error;
}


//Frees the memory associated with dataset.  Returns NULL on success, dataset on failure.
//Fails if dataset is not present in storedResults.
sql_data_t
Db_Query::release(sql_data_t dataset)
{
	//determine if dataset is in storedResults (with sequential search)
	if(dataset == NULL) return dataset;
	
	int result = 0;
	while(result < kMaxResults && storedResults[result] != dataset)
		result ++;

	if(result >= kMaxResults) return dataset;

	//free component strings
	for(int index=0; index < resultLengths[result]; index++)
	{
		if(dataset[index] != NULL)
			delete dataset[index];
	}
	
	//free dataset
	delete dataset;

	//update table
	storedResults[result] = NULL;  
	numResults--;

//	cout << "Dataset released." << endl;

	return NULL;
}	

// check to see if the database is ready to go
bool
Db_Query::ready()
{
	return isValid() && numResults < kMaxResults;
}

// check for an error resulting from a select query
void checkError(sql_data_t sdt, Db_Query *dbq)
{
	if (sdt == NULL)
	{
		cout << dbq->getError();
		exit(1);
	}
}

// check for an error resulting from any non-select query
void checkError(int code, Db_Query *dbq)
{
	if( code==0 )
	{
		cout << dbq->getError();
		exit(1);
	}
}
