/* Use, modification, and distribution are subject to the Boost Software 
License, Version 1.0. (See accompanying file LICENSE_1.0.txt or copy at 
www.boost.org/LICENSE_1.0.txt) */

// Link Class
// Contains information about a web link, such as the link text, the links
// url, a vector of its positions on the page, and a count.

import java.util.*;

/**
 * This class stores information about a web link.
 * @author Aaron Miller, Mike Ottum
 * @version 1.7
 */

public class Link
{
    protected String url;
    protected String sourceUrl;
    protected Vector positions;
    protected HashMap wordList;
    protected int count;

    /**
     * Initializes the data structures for the link by adding the url string
     * and the position.
     * @param myURL The absolute URL that this link points to.
     * @param pos The position of this link on the page.
     */
    public Link(String myURL, String sURL, int pos)
    {
	count = 1;
	url = new String(myURL);
	sourceUrl = new String(sURL);
	positions = new Vector();
	wordList = new HashMap();
	Integer i = new Integer(pos);
	positions.add(i);
    }
    
    /**
     * Tests a link for equality.  The equality test is based on the url
     * string.  Thus, it is important to use absolute url's.
     * @param l The object to compare with the current object.
     * @return True if the objects are equal. False otherwise.
     */
    public boolean equals(Link l)
    {
	return url.equals(l.getURL());
    }
    
    /**
     * Returns the number of distinct positions of the link on the page.
     * @return The size of the positions vector, which is equivalent to the
     * number of times that the link appears on the page.
     */
    public int getCount()
    {
	return count;
    }
    
    /**
     * Returns the url string for the current link.
     * @return The url string for the current link.
     */
    public String getURL()
    {
	return url;
    }
    
    public String getSourceURL()
    {
	return sourceUrl;
    }
    
    /**
     * Add an occurrence of the link on the page.
     * @param pos The position of the link on the page.
     */
    public void addPosition(int pos)
    {
	Integer i = new Integer(pos);
	positions.add(i);
	count++;
    }
    
    /**
     * Add a word associated with this link. These would typically be words
     * enclosed by the &lt;a&gt; and &lt;/a&gt; tags corresponding to the link.
     * @param wordString The word to be added.
     * @param location The location of the word on the page. This location
     * should be absolute rather than relative to the link.
     * @param tags Tag bitstring associated with the word.
     */
    public void addWord(String wordString, int location, String tags)
    {
	Word w;
	if(!wordList.containsKey(wordString))
	    {
		w = new Word(wordString, location);
		wordList.put(wordString, w);
	    }
	else
	    {
		// put the information into the word object that already
		// exists
		w = (Word)(wordList.get(wordString));
	    }
	w.addTags(tags, location);
    }

    /**
     * Returns an {@link Iterator} to the {@link Set} of Words contained in
     * the Link
     */
    public Iterator getWords()
    {
	Iterator it = wordList.keySet().iterator();
	return it;
    }
    
    /**
     * Returns an Iterator to the Vector of integers of positions of the link
     * on the Page.
     */
    public Iterator getPositions()
    {
	return positions.iterator();
    }
}
